/*
 * Decompiled with CFR 0.152.
 */
package haveno.common.file;

import com.google.common.io.Files;
import haveno.common.file.ResourceNotFoundException;
import haveno.common.util.Utilities;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Scanner;
import javax.annotation.Nullable;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileUtil {
    private static final Logger log = LoggerFactory.getLogger(FileUtil.class);
    private static final String BACKUP_DIR = "backup";

    public static void rollingBackup(File dir, String fileName, int numMaxBackupFiles) {
        if (numMaxBackupFiles <= 0) {
            return;
        }
        if (dir.exists()) {
            File origFile;
            File backupDir = new File(Paths.get(dir.getAbsolutePath(), BACKUP_DIR).toString());
            if (!backupDir.exists() && !backupDir.mkdir()) {
                log.warn("make dir failed.\nBackupDir=" + backupDir.getAbsolutePath());
            }
            if ((origFile = new File(Paths.get(dir.getAbsolutePath(), fileName).toString())).exists()) {
                File backupFileDir;
                Object dirName = "backups_" + fileName;
                if (((String)dirName).contains(".")) {
                    dirName = ((String)dirName).replace(".", "_");
                }
                if (!(backupFileDir = new File(Paths.get(backupDir.getAbsolutePath(), new String[]{dirName}).toString())).exists() && !backupFileDir.mkdir()) {
                    log.warn("make backupFileDir failed.\nBackupFileDir=" + backupFileDir.getAbsolutePath());
                }
                File backupFile = new File(Paths.get(backupFileDir.getAbsolutePath(), new Date().getTime() + "_" + fileName).toString());
                try {
                    Files.copy(origFile, backupFile);
                    FileUtil.pruneBackup(backupFileDir, numMaxBackupFiles);
                }
                catch (IOException e) {
                    log.error("Backup key failed: {}\n", (Object)e.getMessage(), (Object)e);
                }
            }
        }
    }

    public static List<File> getBackupFiles(File dir, String fileName) {
        File backupFileDir;
        File backupDir = new File(Paths.get(dir.getAbsolutePath(), BACKUP_DIR).toString());
        if (!backupDir.exists()) {
            return new ArrayList<File>();
        }
        Object dirName = "backups_" + fileName;
        if (((String)dirName).contains(".")) {
            dirName = ((String)dirName).replace(".", "_");
        }
        if (!(backupFileDir = new File(Paths.get(backupDir.getAbsolutePath(), new String[]{dirName}).toString())).exists()) {
            return new ArrayList<File>();
        }
        File[] files = backupFileDir.listFiles();
        return Arrays.asList(files);
    }

    public static File getLatestBackupFile(File dir, String fileName) {
        List<File> files = FileUtil.getBackupFiles(dir, fileName);
        if (files.isEmpty()) {
            return null;
        }
        files.sort(Comparator.comparing(File::getName));
        return files.get(files.size() - 1);
    }

    public static void deleteRollingBackup(File dir, String fileName) {
        File backupDir = new File(Paths.get(dir.getAbsolutePath(), BACKUP_DIR).toString());
        if (!backupDir.exists()) {
            return;
        }
        Object dirName = "backups_" + fileName;
        if (((String)dirName).contains(".")) {
            dirName = ((String)dirName).replace(".", "_");
        }
        File backupFileDir = new File(Paths.get(backupDir.getAbsolutePath(), new String[]{dirName}).toString());
        try {
            FileUtils.deleteDirectory(backupFileDir);
        }
        catch (IOException e) {
            log.error("Delete backup key failed: {}\n", (Object)e.getMessage(), (Object)e);
        }
    }

    private static void pruneBackup(File backupDir, int numMaxBackupFiles) {
        List<File> filesList;
        File[] files;
        if (backupDir.isDirectory() && (files = backupDir.listFiles()) != null && (filesList = Arrays.asList(files)).size() > numMaxBackupFiles) {
            filesList.sort(Comparator.comparing(File::getName));
            File file = filesList.get(0);
            if (file.isFile()) {
                if (!file.delete()) {
                    log.error("Failed to delete file: " + String.valueOf(file));
                } else {
                    FileUtil.pruneBackup(backupDir, numMaxBackupFiles);
                }
            } else {
                FileUtil.pruneBackup(new File(Paths.get(backupDir.getAbsolutePath(), file.getName()).toString()), numMaxBackupFiles);
            }
        }
    }

    public static void deleteDirectory(File file) throws IOException {
        FileUtil.deleteDirectory(file, null, true);
    }

    public static void deleteDirectory(File file, @Nullable File exclude, boolean ignoreLockedFiles) throws IOException {
        File[] files;
        boolean excludeFileFound = false;
        if (file.isDirectory() && (files = file.listFiles()) != null) {
            for (File f : files) {
                boolean excludeFileFoundLocal = exclude != null && f.getAbsolutePath().equals(exclude.getAbsolutePath());
                excludeFileFound |= excludeFileFoundLocal;
                if (excludeFileFoundLocal) continue;
                FileUtil.deleteDirectory(f, exclude, ignoreLockedFiles);
            }
        }
        if (!(excludeFileFound || exclude != null && file.getAbsolutePath().equals(exclude.getAbsolutePath()))) {
            try {
                FileUtil.deleteFileIfExists(file, ignoreLockedFiles);
            }
            catch (Throwable t2) {
                log.error("Could not delete file. Error=" + t2.toString());
                throw new IOException(t2);
            }
        }
    }

    public static void deleteFileIfExists(File file) throws IOException {
        FileUtil.deleteFileIfExists(file, true);
    }

    public static void deleteFileIfExists(File file, boolean ignoreLockedFiles) throws IOException {
        block5: {
            try {
                if (Utilities.isWindows()) {
                    file = file.getCanonicalFile();
                }
                if (!file.exists() || file.delete()) break block5;
                if (ignoreLockedFiles) {
                    if (FileUtil.isFileLocked(file)) {
                        log.info("Failed to delete locked file: " + file.getAbsolutePath());
                    }
                    break block5;
                }
                String message = "Failed to delete file: " + file.getAbsolutePath();
                log.error(message);
                throw new IOException(message);
            }
            catch (Throwable t2) {
                log.error("Could not delete file, error={}\n", (Object)t2.getMessage(), (Object)t2);
                throw new IOException(t2);
            }
        }
    }

    private static boolean isFileLocked(File file) {
        return !file.canWrite();
    }

    public static void resourceToFile(String resourcePath, File destinationFile) throws ResourceNotFoundException, IOException {
        try (InputStream inputStream = ClassLoader.getSystemClassLoader().getResourceAsStream(resourcePath);){
            if (inputStream == null) {
                throw new ResourceNotFoundException(resourcePath);
            }
            try (FileOutputStream fileOutputStream = new FileOutputStream(destinationFile);){
                IOUtils.copy(inputStream, (OutputStream)fileOutputStream);
            }
        }
    }

    public static boolean resourceEqualToFile(String resourcePath, File destinationFile) throws ResourceNotFoundException, IOException {
        try (InputStream inputStream = ClassLoader.getSystemClassLoader().getResourceAsStream(resourcePath);){
            if (inputStream == null) {
                throw new ResourceNotFoundException(resourcePath);
            }
            boolean bl = IOUtils.contentEquals(inputStream, new FileInputStream(destinationFile));
            return bl;
        }
    }

    public static void renameFile(File oldFile, File newFile) throws IOException {
        if (Utilities.isWindows()) {
            File canonical = newFile.getCanonicalFile();
            if (canonical.exists() && !canonical.delete()) {
                throw new IOException("Failed to delete canonical file for replacement with save");
            }
            if (!oldFile.renameTo(canonical)) {
                throw new IOException("Failed to rename " + String.valueOf(oldFile) + " to " + String.valueOf(canonical));
            }
        } else if (!oldFile.renameTo(newFile)) {
            throw new IOException("Failed to rename " + String.valueOf(oldFile) + " to " + String.valueOf(newFile));
        }
    }

    public static void copyFile(File origin, File target) throws IOException {
        if (!origin.exists()) {
            return;
        }
        try {
            Files.copy(origin, target);
        }
        catch (IOException e) {
            log.error("Copy file failed", e);
            throw new IOException("Failed to copy " + String.valueOf(origin) + " to " + String.valueOf(target));
        }
    }

    public static void copyDirectory(File source, File destination) throws IOException {
        FileUtils.copyDirectory(source, destination);
    }

    public static File createNewFile(Path path) throws IOException {
        File file = path.toFile();
        if (!file.createNewFile()) {
            throw new IOException("There already exists a file with path: " + String.valueOf(path));
        }
        return file;
    }

    public static void removeAndBackupFile(File dbDir, File storageFile, String fileName, String backupFolderName) throws IOException {
        File corruptedBackupDir = new File(Paths.get(dbDir.getAbsolutePath(), backupFolderName).toString());
        if (!corruptedBackupDir.exists() && !corruptedBackupDir.mkdir()) {
            log.warn("make dir failed");
        }
        File corruptedFile = new File(Paths.get(dbDir.getAbsolutePath(), backupFolderName, fileName).toString());
        if (storageFile.exists()) {
            FileUtil.renameFile(storageFile, corruptedFile);
        }
    }

    public static boolean doesFileContainKeyword(File file, String keyword) throws FileNotFoundException {
        Scanner s2 = new Scanner(file);
        while (s2.hasNextLine()) {
            if (!s2.nextLine().contains(keyword)) continue;
            return true;
        }
        return false;
    }
}

