/*
 * Decompiled with CFR 0.152.
 */
package haveno.core.api;

import com.google.inject.Inject;
import com.google.inject.Singleton;
import haveno.common.crypto.KeyRing;
import haveno.common.handlers.ErrorMessageHandler;
import haveno.common.handlers.ResultHandler;
import haveno.common.util.MathUtils;
import haveno.core.api.CoreContext;
import haveno.core.api.CoreWalletsService;
import haveno.core.locale.CurrencyUtil;
import haveno.core.locale.Res;
import haveno.core.monetary.Price;
import haveno.core.offer.CreateOfferService;
import haveno.core.offer.Offer;
import haveno.core.offer.OfferBookService;
import haveno.core.offer.OfferDirection;
import haveno.core.offer.OfferFilterService;
import haveno.core.offer.OfferUtil;
import haveno.core.offer.OpenOffer;
import haveno.core.offer.OpenOfferManager;
import haveno.core.payment.PaymentAccount;
import haveno.core.payment.PaymentAccountUtil;
import haveno.core.user.User;
import haveno.core.util.PriceUtil;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.bitcoinj.core.Transaction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class CoreOffersService {
    private static final Logger log = LoggerFactory.getLogger(CoreOffersService.class);
    private final Supplier<Comparator<Offer>> priceComparator = () -> Comparator.comparing(Offer::getPrice);
    private final Supplier<Comparator<OpenOffer>> openOfferPriceComparator = () -> Comparator.comparing(openOffer -> openOffer.getOffer().getPrice());
    private final Supplier<Comparator<Offer>> reversePriceComparator = () -> Comparator.comparing(Offer::getPrice).reversed();
    private final CoreContext coreContext;
    private final KeyRing keyRing;
    private final CoreWalletsService coreWalletsService;
    private final CreateOfferService createOfferService;
    private final OfferBookService offerBookService;
    private final OfferFilterService offerFilter;
    private final OpenOfferManager openOfferManager;
    private final User user;

    @Inject
    public CoreOffersService(CoreContext coreContext, KeyRing keyRing, CoreWalletsService coreWalletsService, CreateOfferService createOfferService, OfferBookService offerBookService, OfferFilterService offerFilter, OpenOfferManager openOfferManager, OfferUtil offerUtil, User user) {
        this.coreContext = coreContext;
        this.keyRing = keyRing;
        this.coreWalletsService = coreWalletsService;
        this.createOfferService = createOfferService;
        this.offerBookService = offerBookService;
        this.offerFilter = offerFilter;
        this.openOfferManager = openOfferManager;
        this.user = user;
    }

    List<Offer> getOffers() {
        List<Offer> offers = new ArrayList<Offer>(this.offerBookService.getOffers()).stream().filter(o -> !o.isMyOffer(this.keyRing)).filter(o -> {
            OfferFilterService.Result result = this.offerFilter.canTakeOffer((Offer)o, this.coreContext.isApiUser());
            return result.isValid() || result == OfferFilterService.Result.HAS_NO_PAYMENT_ACCOUNT_VALID_FOR_OFFER;
        }).collect(Collectors.toList());
        return offers;
    }

    List<Offer> getOffers(String direction, String currencyCode) {
        return this.getOffers().stream().filter(o -> this.offerMatchesDirectionAndCurrency((Offer)o, direction, currencyCode)).sorted(this.priceComparator(direction)).collect(Collectors.toList());
    }

    Offer getOffer(String id) {
        return this.getOffers().stream().filter(o -> o.getId().equals(id)).findAny().orElseThrow(() -> new IllegalStateException(String.format("offer with id '%s' not found", id)));
    }

    List<OpenOffer> getMyOffers() {
        return this.openOfferManager.getOpenOffers().stream().filter(o -> o.getOffer().isMyOffer(this.keyRing)).collect(Collectors.toList());
    }

    List<OpenOffer> getMyOffers(String direction, String currencyCode) {
        return this.getMyOffers().stream().filter(o -> this.offerMatchesDirectionAndCurrency(o.getOffer(), direction, currencyCode)).sorted(this.openOfferPriceComparator(direction, CurrencyUtil.isTraditionalCurrency(currencyCode))).collect(Collectors.toList());
    }

    OpenOffer getMyOffer(String id) {
        return this.openOfferManager.getOpenOffer(id).filter(open -> open.getOffer().isMyOffer(this.keyRing)).orElseThrow(() -> new IllegalStateException(String.format("openoffer with id '%s' not found", id)));
    }

    void postOffer(String currencyCode, String directionAsString, String priceAsString, boolean useMarketBasedPrice, double marketPriceMargin, long amountAsLong, long minAmountAsLong, double securityDepositPct, String triggerPriceAsString, boolean reserveExactAmount, String paymentAccountId, boolean isPrivateOffer, boolean buyerAsTakerWithoutDeposit, String extraInfo, String sourceOfferId, Consumer<Offer> resultHandler, ErrorMessageHandler errorMessageHandler) {
        this.coreWalletsService.verifyWalletsAreAvailable();
        this.coreWalletsService.verifyEncryptedWalletIsUnlocked();
        PaymentAccount paymentAccount = this.user.getPaymentAccount(paymentAccountId);
        if (paymentAccount == null) {
            throw new IllegalArgumentException(String.format("payment account with id %s not found", paymentAccountId));
        }
        if (!sourceOfferId.isEmpty()) {
            this.cloneOffer(sourceOfferId, currencyCode, priceAsString, useMarketBasedPrice, marketPriceMargin, triggerPriceAsString, paymentAccountId, extraInfo, resultHandler, errorMessageHandler);
            return;
        }
        String upperCaseCurrencyCode = currencyCode.toUpperCase();
        String offerId = this.createOfferService.getRandomOfferId();
        OfferDirection direction = OfferDirection.valueOf(directionAsString.toUpperCase());
        Price price = priceAsString.isEmpty() ? null : Price.valueOf(upperCaseCurrencyCode, this.priceStringToLong(priceAsString, upperCaseCurrencyCode));
        BigInteger amount = BigInteger.valueOf(amountAsLong);
        BigInteger minAmount = BigInteger.valueOf(minAmountAsLong);
        Offer offer = this.createOfferService.createAndGetOffer(offerId, direction, upperCaseCurrencyCode, amount, minAmount, price, useMarketBasedPrice, MathUtils.exactMultiply(marketPriceMargin, 0.01), securityDepositPct, paymentAccount, isPrivateOffer, buyerAsTakerWithoutDeposit, extraInfo);
        this.verifyPaymentAccountIsValidForNewOffer(offer, paymentAccount);
        this.placeOffer(offer, triggerPriceAsString, true, reserveExactAmount, null, transaction -> resultHandler.accept(offer), errorMessageHandler);
    }

    private void cloneOffer(String sourceOfferId, String currencyCode, String priceAsString, boolean useMarketBasedPrice, double marketPriceMargin, String triggerPriceAsString, String paymentAccountId, String extraInfo, Consumer<Offer> resultHandler, ErrorMessageHandler errorMessageHandler) {
        PaymentAccount paymentAccount;
        Price price;
        OpenOffer sourceOpenOffer = this.getMyOffer(sourceOfferId);
        Offer sourceOffer = sourceOpenOffer.getOffer();
        if (currencyCode.isEmpty()) {
            currencyCode = sourceOffer.getOfferPayload().getBaseCurrencyCode();
        }
        if (currencyCode.equalsIgnoreCase(Res.getBaseCurrencyCode())) {
            currencyCode = sourceOffer.getOfferPayload().getCounterCurrencyCode();
        }
        String upperCaseCurrencyCode = currencyCode.toUpperCase();
        Object object = useMarketBasedPrice ? null : (priceAsString.isEmpty() ? (sourceOffer.isUseMarketBasedPrice() ? null : sourceOffer.getPrice()) : (price = Price.parse(upperCaseCurrencyCode, priceAsString)));
        if (price == null) {
            useMarketBasedPrice = true;
        }
        if (paymentAccountId.isEmpty()) {
            paymentAccountId = sourceOffer.getOfferPayload().getMakerPaymentAccountId();
        }
        if ((paymentAccount = this.user.getPaymentAccount(paymentAccountId)) == null) {
            throw new IllegalArgumentException(String.format("payment acRcount with id %s not found", paymentAccountId));
        }
        if (extraInfo.isEmpty()) {
            extraInfo = sourceOffer.getOfferPayload().getExtraInfo();
        }
        Offer offer = this.createOfferService.createClonedOffer(sourceOffer, upperCaseCurrencyCode, price, useMarketBasedPrice, MathUtils.exactMultiply(marketPriceMargin, 0.01), paymentAccount, extraInfo);
        this.verifyPaymentAccountIsValidForNewOffer(offer, paymentAccount);
        this.placeOffer(offer, triggerPriceAsString, true, false, sourceOfferId, transaction -> resultHandler.accept(offer), errorMessageHandler);
    }

    Offer editOffer(String offerId, String currencyCode, OfferDirection direction, Price price, boolean useMarketBasedPrice, double marketPriceMargin, BigInteger amount, BigInteger minAmount, double securityDepositPct, PaymentAccount paymentAccount, boolean isPrivateOffer, boolean buyerAsTakerWithoutDeposit, String extraInfo) {
        return this.createOfferService.createAndGetOffer(offerId, direction, currencyCode.toUpperCase(), amount, minAmount, price, useMarketBasedPrice, MathUtils.exactMultiply(marketPriceMargin, 0.01), securityDepositPct, paymentAccount, isPrivateOffer, buyerAsTakerWithoutDeposit, extraInfo);
    }

    void cancelOffer(String id, ResultHandler resultHandler, ErrorMessageHandler errorMessageHandler) {
        Offer offer = this.getMyOffer(id).getOffer();
        this.openOfferManager.removeOffer(offer, resultHandler, errorMessageHandler);
    }

    private void verifyPaymentAccountIsValidForNewOffer(Offer offer, PaymentAccount paymentAccount) {
        if (!PaymentAccountUtil.isPaymentAccountValidForOffer(offer, paymentAccount)) {
            String error = String.format("cannot create %s offer with payment account %s", offer.getOfferPayload().getCounterCurrencyCode(), paymentAccount.getId());
            throw new IllegalStateException(error);
        }
    }

    private void placeOffer(Offer offer, String triggerPriceAsString, boolean useSavingsWallet, boolean reserveExactAmount, String sourceOfferId, Consumer<Transaction> resultHandler, ErrorMessageHandler errorMessageHandler) {
        long triggerPriceAsLong = PriceUtil.getMarketPriceAsLong(triggerPriceAsString, offer.getCurrencyCode());
        this.openOfferManager.placeOffer(offer, useSavingsWallet, triggerPriceAsLong, reserveExactAmount, true, sourceOfferId, resultHandler::accept, errorMessageHandler);
    }

    private boolean offerMatchesDirectionAndCurrency(Offer offer, String direction, String currencyCode) {
        if ("".equals(direction)) {
            direction = null;
        }
        if ("".equals(currencyCode)) {
            currencyCode = null;
        }
        boolean offerOfWantedDirection = direction == null || offer.getDirection().name().equalsIgnoreCase(direction);
        String counterAssetCode = CurrencyUtil.isCryptoCurrency(currencyCode) ? offer.getOfferPayload().getBaseCurrencyCode() : offer.getOfferPayload().getCounterCurrencyCode();
        boolean offerInWantedCurrency = currencyCode == null || counterAssetCode.equalsIgnoreCase(currencyCode);
        return offerOfWantedDirection && offerInWantedCurrency;
    }

    private Comparator<Offer> priceComparator(String direction) {
        return direction.equalsIgnoreCase(OfferDirection.BUY.name()) ? this.reversePriceComparator.get() : this.priceComparator.get();
    }

    private Comparator<OpenOffer> openOfferPriceComparator(String direction, boolean isTraditional) {
        if (isTraditional) {
            return direction.equalsIgnoreCase(OfferDirection.BUY.name()) ? this.openOfferPriceComparator.get().reversed() : this.openOfferPriceComparator.get();
        }
        return direction.equalsIgnoreCase(OfferDirection.SELL.name()) ? this.openOfferPriceComparator.get().reversed() : this.openOfferPriceComparator.get();
    }

    private long priceStringToLong(String priceAsString, String currencyCode) {
        int precision = CurrencyUtil.isTraditionalCurrency(currencyCode) ? 8 : 8;
        double priceAsDouble = new BigDecimal(priceAsString).doubleValue();
        double scaled = MathUtils.scaleUpByPowerOf10(priceAsDouble, precision);
        return MathUtils.roundDoubleToLong(scaled);
    }
}

