/*
 * Decompiled with CFR 0.152.
 */
package haveno.core.api;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.google.inject.name.Named;
import haveno.common.Timer;
import haveno.common.UserThread;
import haveno.core.api.CoreAccountService;
import haveno.core.api.CoreContext;
import haveno.core.api.model.AddressBalanceInfo;
import haveno.core.api.model.BalancesInfo;
import haveno.core.api.model.BtcBalanceInfo;
import haveno.core.api.model.XmrBalanceInfo;
import haveno.core.app.AppStartupState;
import haveno.core.user.Preferences;
import haveno.core.util.ParsingUtils;
import haveno.core.util.coin.CoinFormatter;
import haveno.core.xmr.Balances;
import haveno.core.xmr.model.AddressEntry;
import haveno.core.xmr.setup.WalletsSetup;
import haveno.core.xmr.wallet.BtcWalletService;
import haveno.core.xmr.wallet.Restrictions;
import haveno.core.xmr.wallet.WalletsManager;
import haveno.core.xmr.wallet.XmrWalletService;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import monero.wallet.model.MoneroDestination;
import monero.wallet.model.MoneroTxWallet;
import org.bitcoinj.core.Address;
import org.bitcoinj.core.Coin;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.Transaction;
import org.bitcoinj.core.TransactionConfidence;
import org.bitcoinj.crypto.KeyCrypterScrypt;
import org.bouncycastle.crypto.params.KeyParameter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
class CoreWalletsService {
    private static final Logger log = LoggerFactory.getLogger(CoreWalletsService.class);
    private final AppStartupState appStartupState;
    private final CoreAccountService accountService;
    private final CoreContext coreContext;
    private final Balances balances;
    private final WalletsManager walletsManager;
    private final WalletsSetup walletsSetup;
    private final BtcWalletService btcWalletService;
    private final XmrWalletService xmrWalletService;
    private final CoinFormatter btcFormatter;
    @Nullable
    private Timer lockTimer;
    @Nullable
    private KeyParameter tempAesKey;

    @Inject
    public CoreWalletsService(AppStartupState appStartupState, CoreContext coreContext, CoreAccountService accountService, Balances balances, WalletsManager walletsManager, WalletsSetup walletsSetup, BtcWalletService btcWalletService, XmrWalletService xmrWalletService, @Named(value="BTC") CoinFormatter btcFormatter, Preferences preferences) {
        this.appStartupState = appStartupState;
        this.coreContext = coreContext;
        this.accountService = accountService;
        this.balances = balances;
        this.walletsManager = walletsManager;
        this.walletsSetup = walletsSetup;
        this.btcWalletService = btcWalletService;
        this.xmrWalletService = xmrWalletService;
        this.btcFormatter = btcFormatter;
    }

    @Nullable
    KeyParameter getKey() {
        this.verifyEncryptedWalletIsUnlocked();
        return this.tempAesKey;
    }

    NetworkParameters getNetworkParameters() {
        return this.btcWalletService.getWallet().getContext().getParams();
    }

    BalancesInfo getBalances(String currencyCode) {
        this.accountService.checkAccountOpen();
        this.verifyWalletCurrencyCodeIsValid(currencyCode);
        this.verifyWalletsAreAvailable();
        this.verifyEncryptedWalletIsUnlocked();
        switch (currencyCode.trim().toUpperCase()) {
            case "": 
            case "XMR": {
                return new BalancesInfo(BtcBalanceInfo.EMPTY, this.getXmrBalances());
            }
        }
        throw new IllegalStateException("Unsupported currency code: " + currencyCode.trim().toUpperCase());
    }

    String getXmrSeed() {
        return this.xmrWalletService.getWallet().getSeed();
    }

    String getXmrPrimaryAddress() {
        return this.xmrWalletService.getWallet().getPrimaryAddress();
    }

    String getXmrNewSubaddress() {
        this.accountService.checkAccountOpen();
        return this.xmrWalletService.getNewAddressEntry().getAddressString();
    }

    List<MoneroTxWallet> getXmrTxs() {
        this.accountService.checkAccountOpen();
        return this.xmrWalletService.getTxs();
    }

    MoneroTxWallet createXmrTx(List<MoneroDestination> destinations) {
        this.accountService.checkAccountOpen();
        this.verifyWalletsAreAvailable();
        this.verifyEncryptedWalletIsUnlocked();
        try {
            return this.xmrWalletService.createTx(destinations);
        }
        catch (Exception ex) {
            log.error("", ex);
            throw new IllegalStateException(ex);
        }
    }

    String relayXmrTx(String metadata) {
        this.accountService.checkAccountOpen();
        this.verifyWalletsAreAvailable();
        this.verifyEncryptedWalletIsUnlocked();
        try {
            return this.xmrWalletService.relayTx(metadata);
        }
        catch (Exception ex) {
            log.error("", ex);
            throw new IllegalStateException(ex);
        }
    }

    long getAddressBalance(String addressString) {
        Address address = this.getAddressEntry(addressString).getAddress();
        return this.btcWalletService.getBalanceForAddress((Address)address).value;
    }

    AddressBalanceInfo getAddressBalanceInfo(String addressString) {
        long satoshiBalance = this.getAddressBalance(addressString);
        int numConfirmations = this.getNumConfirmationsForMostRecentTransaction(addressString);
        Address address = this.getAddressEntry(addressString).getAddress();
        return new AddressBalanceInfo(addressString, satoshiBalance, numConfirmations, this.btcWalletService.isAddressUnused(address));
    }

    List<AddressBalanceInfo> getFundingAddresses() {
        this.verifyWalletsAreAvailable();
        this.verifyEncryptedWalletIsUnlocked();
        boolean unusedAddressExists = this.btcWalletService.getAvailableAddressEntries().stream().anyMatch(a -> this.btcWalletService.isAddressUnused(a.getAddress()));
        if (!unusedAddressExists) {
            this.btcWalletService.getFreshAddressEntry();
        }
        List addressStrings = this.btcWalletService.getAvailableAddressEntries().stream().map(AddressEntry::getAddressString).collect(Collectors.toList());
        LoadingCache<String, Long> balances = CoreWalletsService.memoize(this::getAddressBalance);
        boolean noAddressHasZeroBalance = addressStrings.stream().allMatch(addressString -> (Long)balances.getUnchecked((String)addressString) != 0L);
        if (noAddressHasZeroBalance) {
            AddressEntry newZeroBalanceAddress = this.btcWalletService.getFreshAddressEntry();
            addressStrings.add(newZeroBalanceAddress.getAddressString());
        }
        return addressStrings.stream().map(address -> new AddressBalanceInfo((String)address, (Long)balances.getUnchecked((String)address), this.getNumConfirmationsForMostRecentTransaction((String)address), this.btcWalletService.isAddressUnused(this.getAddressEntry((String)address).getAddress()))).collect(Collectors.toList());
    }

    Transaction getTransaction(String txId) {
        if (txId.length() != 64) {
            throw new IllegalArgumentException(String.format("%s is not a transaction id", txId));
        }
        try {
            Transaction tx = this.btcWalletService.getTransaction(txId);
            if (tx == null) {
                throw new IllegalArgumentException(String.format("tx with id %s not found", txId));
            }
            return tx;
        }
        catch (IllegalArgumentException ex) {
            log.error("", ex);
            throw new IllegalArgumentException(String.format("could not get transaction with id %s%ncause: %s", txId, ex.getMessage().toLowerCase()));
        }
    }

    int getNumConfirmationsForMostRecentTransaction(String addressString) {
        Address address = this.getAddressEntry(addressString).getAddress();
        TransactionConfidence confidence = this.btcWalletService.getConfidenceForAddress(address);
        return confidence == null ? 0 : confidence.getDepthInBlocks();
    }

    void setWalletPassword(String password, String newPassword) {
        this.verifyWalletsAreAvailable();
        KeyCrypterScrypt keyCrypterScrypt = this.getKeyCrypterScrypt();
        if (newPassword != null && !newPassword.isEmpty()) {
            if (!this.walletsManager.areWalletsEncrypted()) {
                throw new IllegalStateException("wallet is not encrypted with a password");
            }
            KeyParameter aesKey = keyCrypterScrypt.deriveKey(password);
            if (!this.walletsManager.checkAESKey(aesKey)) {
                throw new IllegalStateException("incorrect old password");
            }
            this.walletsManager.decryptWallets(aesKey);
            aesKey = keyCrypterScrypt.deriveKey(newPassword);
            this.walletsManager.encryptWallets(keyCrypterScrypt, aesKey);
            this.walletsManager.backupWallets();
            return;
        }
        if (this.walletsManager.areWalletsEncrypted()) {
            throw new IllegalStateException("wallet is encrypted with a password");
        }
        KeyParameter aesKey = keyCrypterScrypt.deriveKey(password);
        this.walletsManager.encryptWallets(keyCrypterScrypt, aesKey);
        this.walletsManager.backupWallets();
    }

    void lockWallet() {
        if (!this.walletsManager.areWalletsEncrypted()) {
            throw new IllegalStateException("wallet is not encrypted with a password");
        }
        if (this.tempAesKey == null) {
            throw new IllegalStateException("wallet is already locked");
        }
        this.tempAesKey = null;
    }

    void unlockWallet(String password, long timeout) {
        this.verifyWalletIsAvailableAndEncrypted();
        KeyCrypterScrypt keyCrypterScrypt = this.getKeyCrypterScrypt();
        this.tempAesKey = keyCrypterScrypt.deriveKey(password);
        if (!this.walletsManager.checkAESKey(this.tempAesKey)) {
            throw new IllegalStateException("incorrect password");
        }
        if (this.lockTimer != null) {
            this.lockTimer.stop();
            this.lockTimer = null;
        }
        if (this.coreContext.isApiUser()) {
            this.maybeSetWalletsManagerKey();
        }
        this.lockTimer = UserThread.runAfter(() -> {
            if (this.tempAesKey != null) {
                log.info("Locking wallet after {} second timeout expired.", (Object)timeout);
                this.tempAesKey = null;
            }
        }, timeout, TimeUnit.SECONDS);
    }

    void removeWalletPassword(String password) {
        this.verifyWalletIsAvailableAndEncrypted();
        KeyCrypterScrypt keyCrypterScrypt = this.getKeyCrypterScrypt();
        KeyParameter aesKey = keyCrypterScrypt.deriveKey(password);
        if (!this.walletsManager.checkAESKey(aesKey)) {
            throw new IllegalStateException("incorrect password");
        }
        this.walletsManager.decryptWallets(aesKey);
        this.walletsManager.backupWallets();
    }

    void verifyWalletsAreAvailable() {
        this.verifyWalletAndNetworkIsReady();
        if (!this.walletsManager.areWalletsAvailable()) {
            throw new IllegalStateException("wallet is not yet available");
        }
    }

    void verifyWalletIsAvailableAndEncrypted() {
        this.verifyWalletAndNetworkIsReady();
        if (!this.walletsManager.areWalletsAvailable()) {
            throw new IllegalStateException("wallet is not yet available");
        }
        if (!this.walletsManager.areWalletsEncrypted()) {
            throw new IllegalStateException("wallet is not encrypted with a password");
        }
    }

    void verifyEncryptedWalletIsUnlocked() {
        if (this.walletsManager.areWalletsEncrypted() && !this.accountService.isAccountOpen()) {
            throw new IllegalStateException("wallet is locked");
        }
    }

    void verifyWalletAndNetworkIsReady() {
        if (!this.appStartupState.isWalletAndNetworkReady()) {
            throw new IllegalStateException("wallet and network is not yet initialized");
        }
    }

    void verifyApplicationIsFullyInitialized() {
        if (!this.appStartupState.isApplicationFullyInitialized()) {
            throw new IllegalStateException("server is not fully initialized");
        }
    }

    private void verifyWalletCurrencyCodeIsValid(String currencyCode) {
        if (currencyCode == null || currencyCode.isEmpty()) {
            return;
        }
        if (!currencyCode.equalsIgnoreCase("BTC") && !currencyCode.equalsIgnoreCase("XMR")) {
            throw new IllegalStateException(String.format("wallet does not support %s", currencyCode));
        }
    }

    private void maybeSetWalletsManagerKey() {
        if (this.tempAesKey == null) {
            throw new IllegalStateException("cannot use null key, unlockwallet timeout may have expired");
        }
        if (this.btcWalletService.getAesKey() == null) {
            KeyParameter aesKey = new KeyParameter(this.tempAesKey.getKey());
            this.walletsManager.setAesKey(aesKey);
            this.walletsSetup.getWalletConfig().maybeAddSegwitKeychain(this.walletsSetup.getWalletConfig().btcWallet(), aesKey);
        }
    }

    private XmrBalanceInfo getXmrBalances() {
        this.verifyWalletsAreAvailable();
        this.verifyEncryptedWalletIsUnlocked();
        if (this.balances.getAvailableBalance() == null) {
            throw new IllegalStateException("Balances are not yet available");
        }
        return this.balances.getBalances();
    }

    private Coin getValidTransferAmount(String amount, CoinFormatter coinFormatter) {
        Coin amountAsCoin = ParsingUtils.parseToCoin(amount, coinFormatter);
        if (amountAsCoin.isLessThan(Restrictions.getMinNonDustOutput())) {
            throw new IllegalStateException(String.format("%s is an invalid transfer amount", amount));
        }
        return amountAsCoin;
    }

    private Coin getTxFeeRateFromParamOrPreferenceOrFeeService(String txFeeRate) {
        return txFeeRate.isEmpty() ? this.btcWalletService.getTxFeeForWithdrawalPerVbyte() : Coin.valueOf(Long.parseLong(txFeeRate));
    }

    private KeyCrypterScrypt getKeyCrypterScrypt() {
        KeyCrypterScrypt keyCrypterScrypt = this.walletsManager.getKeyCrypterScrypt();
        if (keyCrypterScrypt == null) {
            throw new IllegalStateException("wallet encrypter is not available");
        }
        return keyCrypterScrypt;
    }

    private AddressEntry getAddressEntry(String addressString) {
        Optional<AddressEntry> addressEntry = this.btcWalletService.getAddressEntryListAsImmutableList().stream().filter(e -> addressString.equals(e.getAddressString())).findFirst();
        if (!addressEntry.isPresent()) {
            throw new IllegalStateException(String.format("address %s not found in wallet", addressString));
        }
        return addressEntry.get();
    }

    private static <I, O> LoadingCache<I, O> memoize(Function<I, O> f) {
        return CacheBuilder.newBuilder().build(CacheLoader.from(f::apply));
    }
}

