/*
 * Decompiled with CFR 0.152.
 */
package haveno.core.filter;

import com.google.common.base.Preconditions;
import com.google.inject.Inject;
import com.google.inject.name.Named;
import haveno.common.app.Version;
import haveno.common.config.Config;
import haveno.common.config.ConfigFileEditor;
import haveno.common.crypto.KeyRing;
import haveno.core.filter.Filter;
import haveno.core.locale.Res;
import haveno.core.payment.payload.PaymentAccountPayload;
import haveno.core.payment.payload.PaymentMethod;
import haveno.core.provider.ProvidersRepository;
import haveno.core.user.Preferences;
import haveno.core.user.User;
import haveno.core.xmr.nodes.XmrNodes;
import haveno.network.p2p.NodeAddress;
import haveno.network.p2p.P2PService;
import haveno.network.p2p.P2PServiceListener;
import haveno.network.p2p.network.BanFilter;
import haveno.network.p2p.storage.HashMapChangedListener;
import haveno.network.p2p.storage.payload.ProtectedStorageEntry;
import java.lang.reflect.Method;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.PublicKey;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Consumer;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javax.annotation.Nullable;
import org.bitcoinj.core.ECKey;
import org.bitcoinj.core.Sha256Hash;
import org.bitcoinj.core.Utils;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FilterManager {
    private static final Logger log = LoggerFactory.getLogger(FilterManager.class);
    private static final String BANNED_PRICE_RELAY_NODES = "bannedPriceRelayNodes";
    private static final String BANNED_SEED_NODES = "bannedSeedNodes";
    private static final String BANNED_XMR_NODES = "bannedXmrNodes";
    private final P2PService p2PService;
    private final KeyRing keyRing;
    private final User user;
    private final Preferences preferences;
    private final ConfigFileEditor configFileEditor;
    private final ProvidersRepository providersRepository;
    private final boolean ignoreDevMsg;
    private final ObjectProperty<Filter> filterProperty = new SimpleObjectProperty<Filter>();
    private final List<Listener> listeners = new CopyOnWriteArrayList<Listener>();
    private final List<String> publicKeys;
    private ECKey filterSigningKey;
    private final Set<Filter> invalidFilters = new HashSet<Filter>();
    private Consumer<String> filterWarningHandler;

    @Inject
    public FilterManager(P2PService p2PService, KeyRing keyRing, User user, Preferences preferences, Config config, ProvidersRepository providersRepository, BanFilter banFilter, @Named(value="ignoreDevMsg") boolean ignoreDevMsg, @Named(value="useDevPrivilegeKeys") boolean useDevPrivilegeKeys) {
        this.p2PService = p2PService;
        this.keyRing = keyRing;
        this.user = user;
        this.preferences = preferences;
        this.configFileEditor = new ConfigFileEditor(config.configFile);
        this.providersRepository = providersRepository;
        this.ignoreDevMsg = ignoreDevMsg;
        this.publicKeys = useDevPrivilegeKeys ? Collections.singletonList("027a381b5333a56e1cc3d90d3a7d07f26509adf7029ed06fc997c656621f8da1ee") : List.of("0358d47858acdc41910325fce266571540681ef83a0d6fedce312bef9810793a27", "029340c3e7d4bb0f9e651b5f590b434fecb6175aeaa57145c7804ff05d210e534f", "034dc7530bf66ffd9580aa98031ea9a18ac2d269f7c56c0e71eca06105b9ed69f9");
        banFilter.setBannedNodePredicate(this::isNodeAddressBannedFromNetwork);
    }

    public void onAllServicesInitialized() {
        if (this.ignoreDevMsg) {
            return;
        }
        this.p2PService.getP2PDataStorage().getMap().values().stream().map(ProtectedStorageEntry::getProtectedStoragePayload).filter(protectedStoragePayload -> protectedStoragePayload instanceof Filter).map(protectedStoragePayload -> (Filter)protectedStoragePayload).forEach(this::onFilterAddedFromNetwork);
        if (Config.baseCurrencyNetwork().isMainnet() && this.getFilter() == null && this.filterWarningHandler != null) {
            this.filterWarningHandler.accept(Res.get("popup.warning.noFilter"));
        }
        this.p2PService.addHashSetChangedListener(new HashMapChangedListener(){

            @Override
            public void onAdded(Collection<ProtectedStorageEntry> protectedStorageEntries) {
                protectedStorageEntries.stream().filter(protectedStorageEntry -> protectedStorageEntry.getProtectedStoragePayload() instanceof Filter).forEach(protectedStorageEntry -> {
                    Filter filter = (Filter)protectedStorageEntry.getProtectedStoragePayload();
                    FilterManager.this.onFilterAddedFromNetwork(filter);
                });
            }

            @Override
            public void onRemoved(Collection<ProtectedStorageEntry> protectedStorageEntries) {
                protectedStorageEntries.stream().filter(protectedStorageEntry -> protectedStorageEntry.getProtectedStoragePayload() instanceof Filter).forEach(protectedStorageEntry -> {
                    Filter filter = (Filter)protectedStorageEntry.getProtectedStoragePayload();
                    FilterManager.this.onFilterRemovedFromNetwork(filter);
                });
            }
        });
        this.p2PService.addP2PServiceListener(new P2PServiceListener(){

            @Override
            public void onDataReceived() {
            }

            @Override
            public void onNoSeedNodeAvailable() {
            }

            @Override
            public void onNoPeersAvailable() {
            }

            @Override
            public void onUpdatedDataReceived() {
                if (FilterManager.this.filterProperty.get() == null) {
                    FilterManager.this.clearBannedNodes();
                }
            }

            @Override
            public void onTorNodeReady() {
            }

            @Override
            public void onHiddenServicePublished() {
            }

            @Override
            public void onSetupFailed(Throwable throwable) {
            }

            @Override
            public void onRequestCustomBridges() {
            }
        });
    }

    public void setFilterWarningHandler(Consumer<String> filterWarningHandler) {
        this.filterWarningHandler = filterWarningHandler;
        this.addListener(filter -> {
            if (filter != null && filterWarningHandler != null) {
                if (filter.getSeedNodes() != null && !filter.getSeedNodes().isEmpty()) {
                    log.info("One of the seed nodes got banned. {}", (Object)filter.getSeedNodes());
                }
                if (filter.getPriceRelayNodes() != null && !filter.getPriceRelayNodes().isEmpty()) {
                    log.info("One of the price relay nodes got banned. {}", (Object)filter.getPriceRelayNodes());
                }
                if (this.requireUpdateToNewVersionForTrading()) {
                    filterWarningHandler.accept(Res.get("popup.warning.mandatoryUpdate.trading"));
                }
            }
        });
    }

    public boolean isPrivilegedDevPubKeyBanned(String pubKeyAsHex) {
        Filter filter = this.getFilter();
        if (filter == null) {
            return false;
        }
        return filter.getBannedPrivilegedDevPubKeys().contains(pubKeyAsHex);
    }

    public boolean canAddDevFilter(String privKeyString) {
        if (privKeyString == null || privKeyString.isEmpty()) {
            return false;
        }
        if (!this.isValidDevPrivilegeKey(privKeyString)) {
            log.warn("Key in invalid");
            return false;
        }
        ECKey ecKeyFromPrivate = this.toECKey(privKeyString);
        String pubKeyAsHex = this.getPubKeyAsHex(ecKeyFromPrivate);
        if (this.isPrivilegedDevPubKeyBanned(pubKeyAsHex)) {
            log.warn("Pub key is banned.");
            return false;
        }
        return true;
    }

    public String getSignerPubKeyAsHex(String privKeyString) {
        ECKey ecKey = this.toECKey(privKeyString);
        return this.getPubKeyAsHex(ecKey);
    }

    public void addDevFilter(Filter filterWithoutSig, String privKeyString) {
        this.setFilterSigningKey(privKeyString);
        String signatureAsBase64 = this.getSignature(filterWithoutSig);
        Filter filterWithSig = Filter.cloneWithSig(filterWithoutSig, signatureAsBase64);
        this.user.setDevelopersFilter(filterWithSig);
        this.p2PService.addProtectedStorageEntry(filterWithSig);
        this.invalidFilters.forEach(filter -> this.removeInvalidFilters((Filter)filter, privKeyString));
    }

    public void addToInvalidFilters(Filter filter) {
        this.invalidFilters.add(filter);
    }

    public void removeInvalidFilters(Filter filter, String privKeyString) {
        if (Arrays.equals(filter.getOwnerPubKey().getEncoded(), this.keyRing.getSignatureKeyPair().getPublic().getEncoded())) {
            log.info("Remove invalid filter {}", (Object)filter);
            this.setFilterSigningKey(privKeyString);
            String signatureAsBase64 = this.getSignature(Filter.cloneWithoutSig(filter));
            Filter filterWithSig = Filter.cloneWithSig(filter, signatureAsBase64);
            boolean result = this.p2PService.removeData(filterWithSig);
            if (!result) {
                log.warn("Could not remove filter {}", (Object)filter);
            }
        } else {
            log.info("The invalid filter is not our own, so we cannot remove it from the network");
        }
    }

    public boolean canRemoveDevFilter(String privKeyString) {
        if (privKeyString == null || privKeyString.isEmpty()) {
            return false;
        }
        Filter developersFilter = this.getDevFilter();
        if (developersFilter == null) {
            log.warn("There is no persisted dev filter to be removed.");
            return false;
        }
        if (!this.isValidDevPrivilegeKey(privKeyString)) {
            log.warn("Key in invalid.");
            return false;
        }
        ECKey ecKeyFromPrivate = this.toECKey(privKeyString);
        String pubKeyAsHex = this.getPubKeyAsHex(ecKeyFromPrivate);
        if (!developersFilter.getSignerPubKeyAsHex().equals(pubKeyAsHex)) {
            log.warn("pubKeyAsHex derived from private key does not match filterSignerPubKey. filterSignerPubKey={}, pubKeyAsHex derived from private key={}", (Object)developersFilter.getSignerPubKeyAsHex(), (Object)pubKeyAsHex);
            return false;
        }
        if (this.isPrivilegedDevPubKeyBanned(pubKeyAsHex)) {
            log.warn("Pub key is banned.");
            return false;
        }
        return true;
    }

    public void removeDevFilter(String privKeyString) {
        this.setFilterSigningKey(privKeyString);
        Filter filterWithSig = this.user.getDevelopersFilter();
        if (filterWithSig == null) {
            return;
        }
        if (this.p2PService.removeData(filterWithSig)) {
            this.user.setDevelopersFilter(null);
        } else {
            log.warn("Removing dev filter from network failed");
        }
    }

    public void addListener(Listener listener) {
        this.listeners.add(listener);
    }

    public ObjectProperty<Filter> filterProperty() {
        return this.filterProperty;
    }

    @Nullable
    public Filter getFilter() {
        return (Filter)this.filterProperty.get();
    }

    @Nullable
    public Filter getDevFilter() {
        return this.user.getDevelopersFilter();
    }

    public PublicKey getOwnerPubKey() {
        return this.keyRing.getSignatureKeyPair().getPublic();
    }

    public boolean isCurrencyBanned(String currencyCode) {
        return this.getFilter() != null && this.getFilter().getBannedCurrencies() != null && this.getFilter().getBannedCurrencies().stream().anyMatch(e -> e.equals(currencyCode));
    }

    public boolean isPaymentMethodBanned(PaymentMethod paymentMethod) {
        return this.getFilter() != null && this.getFilter().getBannedPaymentMethods() != null && this.getFilter().getBannedPaymentMethods().stream().anyMatch(e -> e.equals(paymentMethod.getId()));
    }

    public boolean isOfferIdBanned(String offerId) {
        return this.getFilter() != null && this.getFilter().getBannedOfferIds().stream().anyMatch(e -> e.equals(offerId));
    }

    public boolean isNodeAddressBanned(NodeAddress nodeAddress) {
        return this.getFilter() != null && this.getFilter().getNodeAddressesBannedFromTrading().stream().anyMatch(e -> e.equals(nodeAddress.getFullAddress()));
    }

    public boolean isNodeAddressBannedFromNetwork(NodeAddress nodeAddress) {
        return this.getFilter() != null && this.getFilter().getNodeAddressesBannedFromNetwork().stream().anyMatch(e -> e.equals(nodeAddress.getFullAddress()));
    }

    public boolean isAutoConfExplorerBanned(String address) {
        return this.getFilter() != null && this.getFilter().getBannedAutoConfExplorers().stream().anyMatch(e -> e.equals(address));
    }

    public String getDisableTradeBelowVersion() {
        return this.getFilter() == null || this.getFilter().getDisableTradeBelowVersion() == null || this.getFilter().getDisableTradeBelowVersion().isEmpty() ? null : this.getFilter().getDisableTradeBelowVersion();
    }

    public boolean requireUpdateToNewVersionForTrading() {
        if (this.getFilter() == null) {
            return false;
        }
        boolean requireUpdateToNewVersion = false;
        String getDisableTradeBelowVersion = this.getFilter().getDisableTradeBelowVersion();
        if (getDisableTradeBelowVersion != null && !getDisableTradeBelowVersion.isEmpty()) {
            requireUpdateToNewVersion = Version.isNewVersion(getDisableTradeBelowVersion);
        }
        return requireUpdateToNewVersion;
    }

    public boolean arePeersPaymentAccountDataBanned(PaymentAccountPayload paymentAccountPayload) {
        return this.getFilter() != null && this.getFilter().getBannedPaymentAccounts().stream().filter(paymentAccountFilter -> paymentAccountFilter.getPaymentMethodId().equals(paymentAccountPayload.getPaymentMethodId())).anyMatch(paymentAccountFilter -> {
            try {
                Method method = paymentAccountPayload.getClass().getMethod(paymentAccountFilter.getGetMethodName(), new Class[0]);
                String valueFromInvoke = (String)method.invoke((Object)paymentAccountPayload, new Object[0]);
                return valueFromInvoke.equalsIgnoreCase(paymentAccountFilter.getValue());
            }
            catch (Throwable e) {
                log.error(e.getMessage());
                return false;
            }
        });
    }

    public boolean isWitnessSignerPubKeyBanned(String witnessSignerPubKeyAsHex) {
        return this.getFilter() != null && this.getFilter().getBannedAccountWitnessSignerPubKeys() != null && this.getFilter().getBannedAccountWitnessSignerPubKeys().stream().anyMatch(e -> e.equals(witnessSignerPubKeyAsHex));
    }

    private void onFilterAddedFromNetwork(Filter newFilter) {
        Filter currentFilter = this.getFilter();
        if (!this.isFilterPublicKeyInList(newFilter)) {
            if (newFilter.getSignerPubKeyAsHex() != null && !newFilter.getSignerPubKeyAsHex().isEmpty()) {
                log.warn("isFilterPublicKeyInList failed. Filter.getSignerPubKeyAsHex={}", (Object)newFilter.getSignerPubKeyAsHex());
            } else {
                log.info("isFilterPublicKeyInList failed. Filter.getSignerPubKeyAsHex not set (expected case for pre v1.3.9 filter)");
            }
            return;
        }
        if (!this.isSignatureValid(newFilter)) {
            log.warn("verifySignature failed. Filter={}", (Object)newFilter);
            return;
        }
        if (currentFilter != null) {
            if (currentFilter.getCreationDate() > newFilter.getCreationDate()) {
                log.info("We received a new filter from the network but the creation date is older than the filter we have already. We ignore the new filter.");
                this.addToInvalidFilters(newFilter);
                return;
            }
            log.info("We received a new filter from the network and the creation date is newer than the filter we have already. We ignore the old filter.");
            this.addToInvalidFilters(currentFilter);
            if (this.isPrivilegedDevPubKeyBanned(newFilter.getSignerPubKeyAsHex())) {
                log.warn("Pub key of filter is banned. currentFilter={}, newFilter={}", (Object)currentFilter, (Object)newFilter);
                return;
            }
        }
        this.filterProperty.set(newFilter);
        this.saveBannedNodes(BANNED_SEED_NODES, newFilter.getSeedNodes());
        this.saveBannedNodes(BANNED_XMR_NODES, newFilter.getXmrNodes());
        List<String> priceRelayNodes = newFilter.getPriceRelayNodes();
        this.saveBannedNodes(BANNED_PRICE_RELAY_NODES, priceRelayNodes);
        this.providersRepository.applyBannedNodes(priceRelayNodes);
        if (newFilter.isPreventPublicXmrNetwork() && this.preferences.getMoneroNodesOptionOrdinal() == XmrNodes.MoneroNodesOption.PUBLIC.ordinal()) {
            this.preferences.setMoneroNodesOptionOrdinal(XmrNodes.MoneroNodesOption.PROVIDED.ordinal());
        }
        this.listeners.forEach(e -> e.onFilterAdded(newFilter));
    }

    private void onFilterRemovedFromNetwork(Filter filter) {
        if (!this.isFilterPublicKeyInList(filter)) {
            log.warn("isFilterPublicKeyInList failed. Filter={}", (Object)filter);
            return;
        }
        if (!this.isSignatureValid(filter)) {
            log.warn("verifySignature failed. Filter={}", (Object)filter);
            return;
        }
        if (this.filterProperty.get() != null && !((Filter)this.filterProperty.get()).equals(filter)) {
            return;
        }
        this.clearBannedNodes();
        if (filter.equals(this.user.getDevelopersFilter())) {
            this.user.setDevelopersFilter(null);
        }
        this.filterProperty.set(null);
    }

    private void clearBannedNodes() {
        this.saveBannedNodes(BANNED_XMR_NODES, null);
        this.saveBannedNodes(BANNED_SEED_NODES, null);
        this.saveBannedNodes(BANNED_PRICE_RELAY_NODES, null);
        if (this.providersRepository.getBannedNodes() != null) {
            this.providersRepository.applyBannedNodes(null);
        }
    }

    private void saveBannedNodes(String optionName, List<String> bannedNodes) {
        if (bannedNodes != null) {
            this.configFileEditor.setOption(optionName, String.join((CharSequence)",", bannedNodes));
        } else {
            this.configFileEditor.clearOption(optionName);
        }
    }

    private boolean isValidDevPrivilegeKey(String privKeyString) {
        try {
            ECKey filterSigningKey = this.toECKey(privKeyString);
            String pubKeyAsHex = this.getPubKeyAsHex(filterSigningKey);
            return this.isPublicKeyInList(pubKeyAsHex);
        }
        catch (Throwable t2) {
            return false;
        }
    }

    private void setFilterSigningKey(String privKeyString) {
        this.filterSigningKey = this.toECKey(privKeyString);
    }

    private String getSignature(Filter filterWithoutSig) {
        Sha256Hash hash = this.getSha256Hash(filterWithoutSig);
        ECKey.ECDSASignature ecdsaSignature = this.filterSigningKey.sign(hash);
        byte[] encodeToDER = ecdsaSignature.encodeToDER();
        return new String(Base64.encode(encodeToDER), StandardCharsets.UTF_8);
    }

    private boolean isFilterPublicKeyInList(Filter filter) {
        String signerPubKeyAsHex = filter.getSignerPubKeyAsHex();
        if (!this.isPublicKeyInList(signerPubKeyAsHex)) {
            log.info("Invalid filter (expected case for pre v1.3.9 filter as we still keep that in the network but the new version does not recognize it as valid filter): signerPubKeyAsHex from filter is not part of our pub key list. signerPubKeyAsHex={}, publicKeys={}, filterCreationDate={}", signerPubKeyAsHex, this.publicKeys, new Date(filter.getCreationDate()));
            return false;
        }
        return true;
    }

    private boolean isPublicKeyInList(String pubKeyAsHex) {
        boolean isPublicKeyInList = this.publicKeys.contains(pubKeyAsHex);
        if (!isPublicKeyInList) {
            log.info("pubKeyAsHex is not part of our pub key list (expected case for pre v1.3.9 filter). pubKeyAsHex={}, publicKeys={}", (Object)pubKeyAsHex, (Object)this.publicKeys);
        }
        return isPublicKeyInList;
    }

    private boolean isSignatureValid(Filter filter) {
        try {
            Filter filterForSigVerification = Filter.cloneWithoutSig(filter);
            Sha256Hash hash = this.getSha256Hash(filterForSigVerification);
            Preconditions.checkNotNull(filter.getSignatureAsBase64(), "filter.getSignatureAsBase64() must not be null");
            byte[] sigData = Base64.decode(filter.getSignatureAsBase64());
            ECKey.ECDSASignature ecdsaSignature = ECKey.ECDSASignature.decodeFromDER(sigData);
            String signerPubKeyAsHex = filter.getSignerPubKeyAsHex();
            byte[] decode = Utils.HEX.decode(signerPubKeyAsHex);
            ECKey ecPubKey = ECKey.fromPublicOnly(decode);
            return ecPubKey.verify(hash, ecdsaSignature);
        }
        catch (Throwable e) {
            log.warn("verifySignature failed. filter={}", (Object)filter);
            return false;
        }
    }

    private ECKey toECKey(String privKeyString) {
        return ECKey.fromPrivate(new BigInteger(1, Utils.HEX.decode(privKeyString)));
    }

    private Sha256Hash getSha256Hash(Filter filter) {
        byte[] filterData = filter.toProtoMessage().toByteArray();
        return Sha256Hash.of(filterData);
    }

    private String getPubKeyAsHex(ECKey ecKey) {
        return Utils.HEX.encode(ecKey.getPubKey());
    }

    public static interface Listener {
        public void onFilterAdded(Filter var1);
    }
}

