/*
 * Decompiled with CFR 0.152.
 */
package haveno.core.notifications;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.gson.Gson;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.google.inject.name.Named;
import haveno.common.UserThread;
import haveno.common.util.Utilities;
import haveno.core.notifications.MobileMessage;
import haveno.core.notifications.MobileMessageEncryption;
import haveno.core.notifications.MobileMessageType;
import haveno.core.notifications.MobileModel;
import haveno.core.notifications.MobileNotificationValidator;
import haveno.core.user.Preferences;
import haveno.network.http.HttpClient;
import java.util.UUID;
import java.util.concurrent.Future;
import java.util.function.Consumer;
import javafx.beans.property.BooleanProperty;
import javafx.beans.property.SimpleBooleanProperty;
import org.apache.commons.codec.binary.Hex;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class MobileNotificationService {
    private static final Logger log = LoggerFactory.getLogger(MobileNotificationService.class);
    private static final String SUCCESS = "success";
    private static final String DEV_URL_LOCALHOST = "http://localhost:8080/";
    private static final String DEV_URL = "http://165.227.40.124:8080/";
    private static final String URL = "http://jtboonrvwmq7frkj.onion/";
    private static final String HAVENO_MESSAGE_IOS_MAGIC = "HavenoMessageiOS";
    private static final String HAVENO_MESSAGE_ANDROID_MAGIC = "HavenoMessageAndroid";
    private final Preferences preferences;
    private final MobileMessageEncryption mobileMessageEncryption;
    private final MobileNotificationValidator mobileNotificationValidator;
    private final HttpClient httpClient;
    private final ListeningExecutorService executorService = Utilities.getListeningExecutorService("MobileNotificationService", 10, 15, 600L);
    private final MobileModel mobileModel;
    private boolean setupConfirmationSent;
    private BooleanProperty useSoundProperty = new SimpleBooleanProperty();
    private BooleanProperty useTradeNotificationsProperty = new SimpleBooleanProperty();
    private BooleanProperty useMarketNotificationsProperty = new SimpleBooleanProperty();
    private BooleanProperty usePriceNotificationsProperty = new SimpleBooleanProperty();

    @Inject
    public MobileNotificationService(Preferences preferences, MobileMessageEncryption mobileMessageEncryption, MobileNotificationValidator mobileNotificationValidator, MobileModel mobileModel, HttpClient httpClient, @Named(value="useLocalhostForP2P") boolean useLocalHost) {
        this.preferences = preferences;
        this.mobileMessageEncryption = mobileMessageEncryption;
        this.mobileNotificationValidator = mobileNotificationValidator;
        this.httpClient = httpClient;
        this.mobileModel = mobileModel;
        httpClient.setBaseUrl(useLocalHost ? DEV_URL : URL);
        httpClient.setIgnoreSocks5Proxy(false);
    }

    public void onAllServicesInitialized() {
        String keyAndToken = this.preferences.getPhoneKeyAndToken();
        if (this.mobileNotificationValidator.isValid(keyAndToken)) {
            this.setupConfirmationSent = true;
            this.mobileModel.applyKeyAndToken(keyAndToken);
            this.mobileMessageEncryption.setKey(this.mobileModel.getKey());
        }
        this.useTradeNotificationsProperty.set(this.preferences.isUseTradeNotifications());
        this.useMarketNotificationsProperty.set(this.preferences.isUseMarketNotifications());
        this.usePriceNotificationsProperty.set(this.preferences.isUsePriceNotifications());
        this.useSoundProperty.set(this.preferences.isUseSoundForMobileNotifications());
    }

    public boolean sendMessage(MobileMessage message) throws Exception {
        return this.sendMessage(message, this.useSoundProperty.get());
    }

    public boolean applyKeyAndToken(String keyAndToken) {
        if (this.mobileNotificationValidator.isValid(keyAndToken)) {
            this.mobileModel.applyKeyAndToken(keyAndToken);
            this.mobileMessageEncryption.setKey(this.mobileModel.getKey());
            this.preferences.setPhoneKeyAndToken(keyAndToken);
            if (!this.setupConfirmationSent) {
                try {
                    this.sendConfirmationMessage();
                    this.setupConfirmationSent = true;
                }
                catch (Exception e) {
                    e.printStackTrace();
                }
            }
            return true;
        }
        return false;
    }

    public boolean sendMessage(MobileMessage message, boolean useSound) throws Exception {
        return this.sendMessage(message, useSound, result -> log.debug("sendMessage result=" + result), throwable -> log.error("sendMessage failed. throwable=" + throwable.toString()));
    }

    private boolean sendMessage(MobileMessage message, boolean useSound, Consumer<String> resultHandler, Consumer<Throwable> errorHandler) throws Exception {
        if (this.mobileModel.getKey() == null) {
            return false;
        }
        if (!(switch (message.getMobileMessageType()) {
            case MobileMessageType.SETUP_CONFIRMATION -> true;
            case MobileMessageType.OFFER, MobileMessageType.TRADE, MobileMessageType.DISPUTE -> this.useTradeNotificationsProperty.get();
            case MobileMessageType.PRICE -> this.usePriceNotificationsProperty.get();
            case MobileMessageType.MARKET -> this.useMarketNotificationsProperty.get();
            case MobileMessageType.ERASE -> true;
            default -> false;
        })) {
            return false;
        }
        log.info("Send message: '{}'", (Object)message.getMessage());
        log.info("sendMessage message={}", (Object)message);
        Gson gson = new Gson();
        String json = gson.toJson(message);
        log.info("json " + json);
        StringBuilder padded = new StringBuilder(json);
        while (padded.length() % 16 != 0) {
            padded.append(" ");
        }
        json = padded.toString();
        String uuid = UUID.randomUUID().toString();
        uuid = uuid.replace("-", "");
        String iv = uuid.substring(0, 16);
        String cipher = this.mobileMessageEncryption.encrypt(json, iv);
        log.info("key = " + this.mobileModel.getKey());
        log.info("iv = " + iv);
        log.info("encryptedJson = " + cipher);
        this.doSendMessage(iv, cipher, useSound, resultHandler, errorHandler);
        return true;
    }

    public void sendEraseMessage() throws Exception {
        MobileMessage message = new MobileMessage("", "", MobileMessageType.ERASE);
        this.sendMessage(message, false);
    }

    public void reset() {
        this.mobileModel.reset();
        this.preferences.setPhoneKeyAndToken(null);
        this.setupConfirmationSent = false;
    }

    private void sendConfirmationMessage() throws Exception {
        log.info("sendConfirmationMessage");
        MobileMessage message = new MobileMessage("", "", MobileMessageType.SETUP_CONFIRMATION);
        this.sendMessage(message, true);
    }

    private void doSendMessage(String iv, String cipher, boolean useSound, final Consumer<String> resultHandler, final Consumer<Throwable> errorHandler) throws Exception {
        if (this.httpClient.hasPendingRequest()) {
            log.warn("We have a pending request open. We ignore that request. httpClient {}", (Object)this.httpClient);
            return;
        }
        if (this.mobileModel.getOs() == null) {
            throw new RuntimeException("No mobileModel OS set");
        }
        Object msg = switch (this.mobileModel.getOs()) {
            case MobileModel.OS.IOS -> HAVENO_MESSAGE_IOS_MAGIC;
            case MobileModel.OS.IOS_DEV -> HAVENO_MESSAGE_IOS_MAGIC;
            case MobileModel.OS.ANDROID -> HAVENO_MESSAGE_ANDROID_MAGIC;
            default -> throw new RuntimeException("No mobileModel OS set");
        };
        msg = (String)msg + "|" + iv + "|" + cipher;
        boolean isAndroid = this.mobileModel.getOs() == MobileModel.OS.ANDROID;
        boolean isProduction = this.mobileModel.getOs() == MobileModel.OS.IOS;
        Preconditions.checkNotNull(this.mobileModel.getToken(), "mobileModel.getToken() must not be null");
        String tokenAsHex = Hex.encodeHexString(this.mobileModel.getToken().getBytes("UTF-8"));
        String msgAsHex = Hex.encodeHexString(((String)msg).getBytes("UTF-8"));
        String param = "relay?isAndroid=" + isAndroid + "&isProduction=" + isProduction + "&isContentAvailable=" + this.mobileModel.isContentAvailable() + "&snd=" + useSound + "&token=" + tokenAsHex + "&msg=" + msgAsHex;
        log.info("Send: token={}", (Object)this.mobileModel.getToken());
        log.info("Send: msg={}", msg);
        log.info("Send: isAndroid={}\nuseSound={}\ntokenAsHex={}\nmsgAsHex={}", isAndroid, useSound, tokenAsHex, msgAsHex);
        String threadName = "sendMobileNotification-" + msgAsHex.substring(0, 5) + "...";
        Future future = this.executorService.submit(() -> {
            Thread.currentThread().setName(threadName);
            String result = this.httpClient.get(param, "User-Agent", "haveno/1.1.1, uid:" + this.httpClient.getUid());
            log.info("sendMobileNotification result: " + result);
            Preconditions.checkArgument(result.equals(SUCCESS), "Result was not 'success'. result=" + result);
            return result;
        });
        Futures.addCallback(future, new FutureCallback<String>(){

            @Override
            public void onSuccess(String result) {
                UserThread.execute(() -> resultHandler.accept(result));
            }

            @Override
            public void onFailure(@NotNull Throwable throwable) {
                UserThread.execute(() -> errorHandler.accept(throwable));
            }
        }, MoreExecutors.directExecutor());
    }

    public MobileModel getMobileModel() {
        return this.mobileModel;
    }

    public boolean isSetupConfirmationSent() {
        return this.setupConfirmationSent;
    }

    public BooleanProperty getUseSoundProperty() {
        return this.useSoundProperty;
    }

    public BooleanProperty getUseTradeNotificationsProperty() {
        return this.useTradeNotificationsProperty;
    }

    public BooleanProperty getUseMarketNotificationsProperty() {
        return this.useMarketNotificationsProperty;
    }

    public BooleanProperty getUsePriceNotificationsProperty() {
        return this.usePriceNotificationsProperty;
    }
}

