/*
 * Decompiled with CFR 0.152.
 */
package haveno.network.p2p.peers;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.inject.Inject;
import com.google.inject.name.Named;
import haveno.common.ClockWatcher;
import haveno.common.Timer;
import haveno.common.UserThread;
import haveno.common.app.Capabilities;
import haveno.common.app.Capability;
import haveno.common.persistence.PersistenceManager;
import haveno.common.proto.persistable.PersistedDataHost;
import haveno.network.p2p.NodeAddress;
import haveno.network.p2p.network.CloseConnectionReason;
import haveno.network.p2p.network.Connection;
import haveno.network.p2p.network.ConnectionListener;
import haveno.network.p2p.network.InboundConnection;
import haveno.network.p2p.network.NetworkNode;
import haveno.network.p2p.network.PeerType;
import haveno.network.p2p.network.RuleViolation;
import haveno.network.p2p.peers.peerexchange.Peer;
import haveno.network.p2p.peers.peerexchange.PeerList;
import haveno.network.p2p.seed.SeedNodeRepository;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class PeerManager
implements ConnectionListener,
PersistedDataHost {
    private static final Logger log = LoggerFactory.getLogger(PeerManager.class);
    private static final long CHECK_MAX_CONN_DELAY_SEC = 10L;
    private static final long REMOVE_ANONYMOUS_PEER_SEC = 240L;
    private static final int MAX_REPORTED_PEERS = 1000;
    private static final int MAX_PERSISTED_PEERS = 500;
    private static final long MAX_AGE = TimeUnit.DAYS.toMillis(14L);
    private static final long MAX_AGE_LIVE_PEERS = TimeUnit.MINUTES.toMillis(30L);
    private static final boolean PRINT_REPORTED_PEERS_DETAILS = true;
    private Timer printStatisticsTimer;
    private boolean shutDownRequested;
    private int numOnConnections;
    private final NetworkNode networkNode;
    private final ClockWatcher clockWatcher;
    private final Set<NodeAddress> seedNodeAddresses;
    private final PersistenceManager<PeerList> persistenceManager;
    private final ClockWatcher.Listener clockWatcherListener;
    private final List<Listener> listeners = new CopyOnWriteArrayList<Listener>();
    private final PeerList peerList = new PeerList();
    private final Set<Peer> reportedPeers = new HashSet<Peer>();
    private final Set<Peer> latestLivePeers = new HashSet<Peer>();
    private Timer checkMaxConnectionsTimer;
    private boolean stopped;
    private boolean lostAllConnections;
    private int maxConnections;
    private int minConnections;
    private int outBoundPeerTrigger;
    private int initialDataExchangeTrigger;
    private int maxConnectionsAbsolute;
    private int peakNumConnections;
    private int numAllConnectionsLostEvents;

    @Inject
    public PeerManager(NetworkNode networkNode, SeedNodeRepository seedNodeRepository, ClockWatcher clockWatcher, PersistenceManager<PeerList> persistenceManager, @Named(value="maxConnections") int maxConnections) {
        this.networkNode = networkNode;
        this.seedNodeAddresses = new HashSet<NodeAddress>(seedNodeRepository.getSeedNodeAddresses());
        this.clockWatcher = clockWatcher;
        this.persistenceManager = persistenceManager;
        this.persistenceManager.initialize(this.peerList, PersistenceManager.Source.PRIVATE_LOW_PRIO);
        this.networkNode.addConnectionListener(this);
        this.setConnectionLimits(maxConnections);
        this.clockWatcherListener = new ClockWatcher.Listener(){

            @Override
            public void onSecondTick() {
            }

            @Override
            public void onMinuteTick() {
            }

            @Override
            public void onAwakeFromStandby(long missedMs) {
                PeerManager.this.stopped = false;
                PeerManager.this.listeners.forEach(Listener::onAwakeFromStandby);
            }
        };
        clockWatcher.addListener(this.clockWatcherListener);
        this.printStatisticsTimer = UserThread.runPeriodically(this::printStatistics, TimeUnit.MINUTES.toSeconds(60L));
    }

    public void shutDown() {
        this.shutDownRequested = true;
        this.networkNode.removeConnectionListener(this);
        this.clockWatcher.removeListener(this.clockWatcherListener);
        this.stopCheckMaxConnectionsTimer();
        if (this.printStatisticsTimer != null) {
            this.printStatisticsTimer.stop();
            this.printStatisticsTimer = null;
        }
    }

    @Override
    public void readPersisted(Runnable completeHandler) {
        this.persistenceManager.readPersisted(persisted -> {
            this.peerList.setAll(persisted.getSet());
            completeHandler.run();
        }, completeHandler);
    }

    @Override
    public void onConnection(Connection connection) {
        connection.getConnectionState().setSeedNode(this.isSeedNode(connection));
        this.doHouseKeeping();
        ++this.numOnConnections;
        if (this.lostAllConnections) {
            this.lostAllConnections = false;
            this.stopped = false;
            log.info("\n------------------------------------------------------------\nEstablished a new connection from/to {} after all connections lost.\n------------------------------------------------------------", (Object)connection.getPeersNodeAddressOptional());
            this.listeners.forEach(Listener::onNewConnectionAfterAllConnectionsLost);
        }
        connection.getPeersNodeAddressOptional().flatMap(this::findPeer).ifPresent(Peer::onConnection);
    }

    @Override
    public void onDisconnect(CloseConnectionReason closeConnectionReason, Connection connection) {
        log.debug("onDisconnect called: nodeAddress={}, closeConnectionReason={}", (Object)connection.getPeersNodeAddressOptional(), (Object)closeConnectionReason);
        this.handleConnectionFault(connection);
        boolean previousLostAllConnections = this.lostAllConnections;
        this.lostAllConnections = this.networkNode.getAllConnections().isEmpty();
        if (this.lostAllConnections && this.numOnConnections > 2) {
            this.stopped = true;
            if (!this.shutDownRequested) {
                if (!previousLostAllConnections) {
                    ++this.numAllConnectionsLostEvents;
                }
                log.warn("\n------------------------------------------------------------\nAll connections lost\n------------------------------------------------------------");
                this.listeners.forEach(Listener::onAllConnectionsLost);
            }
        }
        this.maybeRemoveBannedPeer(closeConnectionReason, connection);
    }

    public boolean hasSufficientConnections() {
        return this.networkNode.getConfirmedConnections().size() >= this.minConnections;
    }

    public boolean isConfirmed(NodeAddress nodeAddress) {
        return this.networkNode.getNodeAddressesOfConfirmedConnections().contains(nodeAddress);
    }

    public void handleConnectionFault(Connection connection) {
        connection.getPeersNodeAddressOptional().ifPresent(nodeAddress -> this.handleConnectionFault((NodeAddress)nodeAddress, connection));
    }

    public void handleConnectionFault(NodeAddress nodeAddress) {
        this.handleConnectionFault(nodeAddress, null);
    }

    public void handleConnectionFault(NodeAddress nodeAddress, @Nullable Connection connection) {
        boolean doRemovePersistedPeer = false;
        this.removeReportedPeer(nodeAddress);
        Optional<Peer> persistedPeerOptional = this.findPersistedPeer(nodeAddress);
        if (persistedPeerOptional.isPresent()) {
            Peer persistedPeer = persistedPeerOptional.get();
            persistedPeer.onDisconnect();
            doRemovePersistedPeer = persistedPeer.tooManyFailedConnectionAttempts();
        }
        boolean ruleViolation = connection != null && connection.getRuleViolation() != null;
        boolean bl = doRemovePersistedPeer = doRemovePersistedPeer || ruleViolation;
        if (doRemovePersistedPeer) {
            this.removePersistedPeer(nodeAddress);
        } else {
            this.removeTooOldPersistedPeers();
        }
    }

    public boolean isSeedNode(Connection connection) {
        return connection.getPeersNodeAddressOptional().isPresent() && this.isSeedNode(connection.getPeersNodeAddressOptional().get());
    }

    public boolean isSelf(NodeAddress nodeAddress) {
        return nodeAddress.equals(this.networkNode.getNodeAddress());
    }

    private boolean isSeedNode(Peer peer) {
        return this.seedNodeAddresses.contains(peer.getNodeAddress());
    }

    public boolean isSeedNode(NodeAddress nodeAddress) {
        return this.seedNodeAddresses.contains(nodeAddress);
    }

    public boolean isPeerBanned(CloseConnectionReason closeConnectionReason, Connection connection) {
        return closeConnectionReason == CloseConnectionReason.PEER_BANNED && connection.getPeersNodeAddressOptional().isPresent();
    }

    private void maybeRemoveBannedPeer(CloseConnectionReason closeConnectionReason, Connection connection) {
        if (connection.getPeersNodeAddressOptional().isPresent() && this.isPeerBanned(closeConnectionReason, connection)) {
            NodeAddress nodeAddress = connection.getPeersNodeAddressOptional().get();
            this.seedNodeAddresses.remove(nodeAddress);
            this.removePersistedPeer(nodeAddress);
            this.removeReportedPeer(nodeAddress);
        }
    }

    public void maybeResetNumAllConnectionsLostEvents() {
        if (!this.networkNode.getAllConnections().isEmpty()) {
            this.numAllConnectionsLostEvents = 0;
        }
    }

    public Optional<Peer> findPeer(NodeAddress peersNodeAddress) {
        return this.getAllPeers().stream().filter(peer -> peer.getNodeAddress().equals(peersNodeAddress)).findAny();
    }

    public Set<Peer> getAllPeers() {
        HashSet<Peer> allPeers = new HashSet<Peer>(this.getLivePeers());
        allPeers.addAll(this.getPersistedPeers());
        allPeers.addAll(this.reportedPeers);
        return allPeers;
    }

    public Collection<Peer> getPersistedPeers() {
        return this.peerList.getSet();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addToReportedPeers(Set<Peer> reportedPeersToAdd, Connection connection, Capabilities capabilities) {
        this.applyCapabilities(connection, capabilities);
        Set<Peer> peers = reportedPeersToAdd.stream().filter(peer -> !this.isSelf(peer.getNodeAddress())).collect(Collectors.toSet());
        this.printNewReportedPeers(peers);
        if (peers.size() <= 1000 + this.maxConnectionsAbsolute + 10) {
            Set<Peer> set = this.reportedPeers;
            synchronized (set) {
                this.reportedPeers.addAll(peers);
                this.purgeReportedPeersIfExceeds();
                this.getPersistedPeers().addAll(peers);
                this.purgePersistedPeersIfExceeds();
                this.requestPersistence();
                this.printReportedPeers();
            }
        } else {
            connection.reportInvalidRequest(RuleViolation.TOO_MANY_REPORTED_PEERS_SENT, "Too many reported peers sent");
        }
    }

    public Set<Peer> getLivePeers() {
        return this.getLivePeers(null);
    }

    public Set<Peer> getLivePeers(@Nullable NodeAddress excludedNodeAddress) {
        int oldNumLatestLivePeers = this.latestLivePeers.size();
        HashSet<Peer> peers = new HashSet<Peer>(this.latestLivePeers);
        Set currentLivePeers = this.getConnectedReportedPeers().stream().filter(e -> !this.isSeedNode((Peer)e)).filter(e -> !e.getNodeAddress().equals(excludedNodeAddress)).collect(Collectors.toSet());
        peers.addAll(currentLivePeers);
        long maxAge = new Date().getTime() - MAX_AGE_LIVE_PEERS;
        this.latestLivePeers.clear();
        Set recentPeers = peers.stream().filter(peer -> peer.getDateAsLong() > maxAge).collect(Collectors.toSet());
        this.latestLivePeers.addAll(recentPeers);
        if (oldNumLatestLivePeers != this.latestLivePeers.size()) {
            log.info("Num of latestLivePeers={}", (Object)this.latestLivePeers.size());
        }
        return this.latestLivePeers;
    }

    public boolean peerHasCapability(NodeAddress peersNodeAddress, Capability capability) {
        return this.findPeersCapabilities(peersNodeAddress).map(capabilities -> capabilities.contains(capability)).orElse(false);
    }

    public Optional<Capabilities> findPeersCapabilities(NodeAddress nodeAddress) {
        Optional<Capabilities> optionalCapabilities = this.networkNode.findPeersCapabilities(nodeAddress);
        if (optionalCapabilities.isPresent() && !optionalCapabilities.get().isEmpty()) {
            return optionalCapabilities;
        }
        return this.getAllPeers().stream().filter(peer -> peer.getNodeAddress().equals(nodeAddress)).findAny().map(Peer::getCapabilities);
    }

    private void applyCapabilities(Connection connection, Capabilities newCapabilities) {
        if (newCapabilities == null || newCapabilities.isEmpty()) {
            return;
        }
        connection.getPeersNodeAddressOptional().ifPresent(nodeAddress -> this.getAllPeers().stream().filter(peer -> peer.getNodeAddress().equals(nodeAddress)).filter(peer -> peer.getCapabilities().hasLess(newCapabilities)).forEach(peer -> peer.setCapabilities(newCapabilities)));
        this.requestPersistence();
    }

    private void doHouseKeeping() {
        if (this.checkMaxConnectionsTimer == null) {
            this.printConnectedPeers();
            this.checkMaxConnectionsTimer = UserThread.runAfter(() -> {
                this.stopCheckMaxConnectionsTimer();
                if (!this.stopped) {
                    HashSet<Connection> allConnections = new HashSet<Connection>(this.networkNode.getAllConnections());
                    int size = allConnections.size();
                    this.peakNumConnections = Math.max(this.peakNumConnections, size);
                    this.removeAnonymousPeers();
                    this.removeTooOldReportedPeers();
                    this.removeTooOldPersistedPeers();
                    this.checkMaxConnections();
                } else {
                    log.debug("We have stopped already. We ignore that checkMaxConnectionsTimer.run call.");
                }
            }, 10L);
        }
    }

    @VisibleForTesting
    boolean checkMaxConnections() {
        HashSet<Connection> allConnections = new HashSet<Connection>(this.networkNode.getAllConnections());
        int size = allConnections.size();
        log.info("We have {} connections open. Our limit is {}", (Object)size, (Object)this.maxConnections);
        if (size <= this.maxConnections) {
            log.debug("We have not exceeded the maxConnections limit of {} so don't need to close any connections.", (Object)size);
            return false;
        }
        log.info("We have too many connections open. Lets try first to remove the inbound connections of type PEER.");
        List candidates = allConnections.stream().filter(e -> e instanceof InboundConnection).filter(e -> e.getConnectionState().getPeerType() == PeerType.PEER).sorted(Comparator.comparingLong(o -> o.getStatistic().getLastActivityTimestamp())).collect(Collectors.toList());
        if (candidates.isEmpty()) {
            log.info("No candidates found. We check if we exceed our outBoundPeerTrigger of {}", (Object)this.outBoundPeerTrigger);
            if (size <= this.outBoundPeerTrigger) {
                log.info("We have not exceeded outBoundPeerTrigger of {} so don't need to close any connections", (Object)this.outBoundPeerTrigger);
                return false;
            }
            log.info("We have exceeded outBoundPeerTrigger of {}. Lets try to remove outbound connection of type PEER.", (Object)this.outBoundPeerTrigger);
            candidates = allConnections.stream().filter(e -> e.getConnectionState().getPeerType() == PeerType.PEER).sorted(Comparator.comparingLong(o -> o.getStatistic().getLastActivityTimestamp())).collect(Collectors.toList());
            if (candidates.isEmpty()) {
                log.info("No candidates found. We check if we exceed our initialDataExchangeTrigger of {}", (Object)this.initialDataExchangeTrigger);
                if (size <= this.initialDataExchangeTrigger) {
                    log.info("We have not exceeded initialDataExchangeTrigger of {} so don't need to close any connections", (Object)this.initialDataExchangeTrigger);
                    return false;
                }
                log.info("We have exceeded initialDataExchangeTrigger of {} Lets try to remove the oldest INITIAL_DATA_EXCHANGE connection.", (Object)this.initialDataExchangeTrigger);
                candidates = allConnections.stream().filter(e -> e.getConnectionState().getPeerType() == PeerType.INITIAL_DATA_EXCHANGE).sorted(Comparator.comparingLong(o -> o.getConnectionState().getLastInitialDataMsgTimeStamp())).collect(Collectors.toList());
                if (candidates.isEmpty()) {
                    log.info("No candidates found. We check if we exceed our maxConnectionsAbsolute limit of {}", (Object)this.maxConnectionsAbsolute);
                    if (size <= this.maxConnectionsAbsolute) {
                        log.info("We have not exceeded maxConnectionsAbsolute limit of {} so don't need to close any connections", (Object)this.maxConnectionsAbsolute);
                        return false;
                    }
                    log.info("We reached abs. max. connections. Lets try to remove ANY connection.");
                    candidates = allConnections.stream().sorted(Comparator.comparingLong(o -> o.getStatistic().getLastActivityTimestamp())).collect(Collectors.toList());
                }
            }
        }
        if (!candidates.isEmpty()) {
            Connection connection = (Connection)candidates.remove(0);
            log.info("checkMaxConnections: Num candidates (inbound/peer) for shut down={}. We close oldest connection to peer {}", (Object)candidates.size(), (Object)connection.getPeersNodeAddressOptional());
            if (!connection.isStopped()) {
                connection.shutDown(CloseConnectionReason.TOO_MANY_CONNECTIONS_OPEN, () -> UserThread.runAfter(this::checkMaxConnections, 100L, TimeUnit.MILLISECONDS));
                return true;
            }
        }
        log.info("No candidates found to remove. size={}, allConnections={}", (Object)size, (Object)allConnections);
        return false;
    }

    private void removeAnonymousPeers() {
        this.networkNode.getAllConnections().stream().filter(connection -> !connection.hasPeersNodeAddress()).filter(connection -> connection.getConnectionState().getPeerType() == PeerType.PEER).forEach(connection -> UserThread.runAfter(() -> {
            if (!connection.isStopped() && !connection.hasPeersNodeAddress()) {
                log.info("removeAnonymousPeers: We close the connection as the peer address is still unknown. Peer: {}", (Object)connection.getPeersNodeAddressOptional());
                connection.shutDown(CloseConnectionReason.UNKNOWN_PEER_ADDRESS);
            }
        }, 240L));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void removeReportedPeer(Peer reportedPeer) {
        Set<Peer> set = this.reportedPeers;
        synchronized (set) {
            this.reportedPeers.remove(reportedPeer);
            this.printReportedPeers();
        }
    }

    private void removeReportedPeer(NodeAddress nodeAddress) {
        ArrayList<Peer> reportedPeersClone = new ArrayList<Peer>(this.reportedPeers);
        reportedPeersClone.stream().filter(e -> e.getNodeAddress().equals(nodeAddress)).findAny().ifPresent(this::removeReportedPeer);
    }

    private void removeTooOldReportedPeers() {
        ArrayList<Peer> reportedPeersClone = new ArrayList<Peer>(this.reportedPeers);
        Set<Peer> reportedPeersToRemove = reportedPeersClone.stream().filter(reportedPeer -> new Date().getTime() - reportedPeer.getDate().getTime() > MAX_AGE).collect(Collectors.toSet());
        reportedPeersToRemove.forEach(this::removeReportedPeer);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void purgeReportedPeersIfExceeds() {
        Set<Peer> set = this.reportedPeers;
        synchronized (set) {
            int size = this.reportedPeers.size();
            if (size > 1000) {
                log.info("We have already {} reported peers which exceeds our limit of {}.We remove random peers from the reported peers list.", (Object)size, (Object)1000);
                int diff = size - 1000;
                ArrayList<Peer> list = new ArrayList<Peer>(this.reportedPeers);
                for (int i = 0; i < diff; ++i) {
                    if (list.isEmpty()) continue;
                    Peer toRemove = (Peer)list.remove(new Random().nextInt(list.size()));
                    this.removeReportedPeer(toRemove);
                }
            } else {
                log.trace("No need to purge reported peers.\n\tWe don't have more then {} reported peers yet.", (Object)1000);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void printReportedPeers() {
        Set<Peer> set = this.reportedPeers;
        synchronized (set) {
            if (!this.reportedPeers.isEmpty()) {
                StringBuilder result = new StringBuilder("\n\n------------------------------------------------------------\nCollected reported peers:");
                ArrayList<Peer> reportedPeersClone = new ArrayList<Peer>(this.reportedPeers);
                reportedPeersClone.forEach(e -> result.append("\n").append(e));
                result.append("\n------------------------------------------------------------\n");
                log.trace(result.toString());
                log.debug("Number of reported peers: {}", (Object)this.reportedPeers.size());
            }
        }
    }

    private void printNewReportedPeers(Set<Peer> reportedPeers) {
        StringBuilder result = new StringBuilder("We received new reportedPeers:");
        ArrayList<Peer> reportedPeersClone = new ArrayList<Peer>(reportedPeers);
        reportedPeersClone.forEach(e -> result.append("\n\t").append(e));
        log.trace(result.toString());
        log.debug("Number of new arrived reported peers: {}", (Object)reportedPeers.size());
    }

    private boolean removePersistedPeer(Peer persistedPeer) {
        if (this.getPersistedPeers().contains(persistedPeer)) {
            this.getPersistedPeers().remove(persistedPeer);
            this.requestPersistence();
            return true;
        }
        return false;
    }

    private void requestPersistence() {
        this.persistenceManager.requestPersistence();
    }

    private boolean removePersistedPeer(NodeAddress nodeAddress) {
        Optional<Peer> optionalPersistedPeer = this.findPersistedPeer(nodeAddress);
        return optionalPersistedPeer.isPresent() && this.removePersistedPeer(optionalPersistedPeer.get());
    }

    private Optional<Peer> findPersistedPeer(NodeAddress nodeAddress) {
        return this.getPersistedPeers().stream().filter(e -> e.getNodeAddress().equals(nodeAddress)).findAny();
    }

    private void removeTooOldPersistedPeers() {
        Set<Peer> persistedPeersToRemove = this.getPersistedPeers().stream().filter(reportedPeer -> new Date().getTime() - reportedPeer.getDate().getTime() > MAX_AGE).collect(Collectors.toSet());
        persistedPeersToRemove.forEach(this::removePersistedPeer);
    }

    private void purgePersistedPeersIfExceeds() {
        int limit;
        int size = this.getPersistedPeers().size();
        if (size > (limit = 500)) {
            log.trace("We have already {} persisted peers which exceeds our limit of {}.We remove random peers from the persisted peers list.", (Object)size, (Object)limit);
            int diff = size - limit;
            ArrayList<Peer> list = new ArrayList<Peer>(this.getPersistedPeers());
            for (int i = 0; i < diff; ++i) {
                if (list.isEmpty()) continue;
                Peer toRemove = (Peer)list.remove(new Random().nextInt(list.size()));
                this.removePersistedPeer(toRemove);
            }
        } else {
            log.trace("No need to purge persisted peers.\n\tWe don't have more then {} persisted peers yet.", (Object)500);
        }
    }

    public int getMaxConnections() {
        return this.maxConnections;
    }

    public void addListener(Listener listener) {
        this.listeners.add(listener);
    }

    public void removeListener(Listener listener) {
        this.listeners.remove(listener);
    }

    private void setConnectionLimits(int maxConnections) {
        this.maxConnections = maxConnections;
        this.minConnections = Math.max(1, (int)Math.round((double)maxConnections * 0.7));
        this.outBoundPeerTrigger = Math.max(4, (int)Math.round((double)maxConnections * 1.3));
        this.initialDataExchangeTrigger = Math.max(8, (int)Math.round((double)maxConnections * 1.7));
        this.maxConnectionsAbsolute = Math.max(12, (int)Math.round((double)maxConnections * 2.5));
    }

    private Set<Peer> getConnectedReportedPeers() {
        return this.networkNode.getConfirmedConnections().stream().map(connection -> {
            Capabilities supportedCapabilities = new Capabilities(connection.getCapabilities());
            Optional<NodeAddress> peersNodeAddressOptional = connection.getPeersNodeAddressOptional();
            Preconditions.checkArgument(peersNodeAddressOptional.isPresent());
            NodeAddress peersNodeAddress = peersNodeAddressOptional.get();
            boolean capabilitiesNotFoundInConnection = supportedCapabilities.isEmpty();
            if (capabilitiesNotFoundInConnection) {
                HashSet<Peer> persistedAndReported = new HashSet<Peer>(this.getPersistedPeers());
                Set<Peer> set = this.reportedPeers;
                synchronized (set) {
                    persistedAndReported.addAll(this.reportedPeers);
                }
                Optional<Peer> candidate = persistedAndReported.stream().filter(peer -> peer.getNodeAddress().equals(peersNodeAddress)).filter(peer -> !peer.getCapabilities().isEmpty()).findAny();
                if (candidate.isPresent()) {
                    supportedCapabilities = new Capabilities(candidate.get().getCapabilities());
                }
            }
            Peer peer2 = new Peer(peersNodeAddress, supportedCapabilities);
            if (capabilitiesNotFoundInConnection) {
                connection.addWeakCapabilitiesListener(peer2);
            }
            return peer2;
        }).collect(Collectors.toSet());
    }

    private void stopCheckMaxConnectionsTimer() {
        if (this.checkMaxConnectionsTimer != null) {
            this.checkMaxConnectionsTimer.stop();
            this.checkMaxConnectionsTimer = null;
        }
    }

    private void printStatistics() {
        String ls = System.lineSeparator();
        StringBuilder sb = new StringBuilder("Connection statistics: " + ls);
        AtomicInteger counter = new AtomicInteger();
        this.networkNode.getAllConnections().stream().sorted(Comparator.comparingLong(o -> o.getConnectionStatistics().getConnectionCreationTimeStamp())).forEach(e -> sb.append(ls).append("Connection ").append(counter.incrementAndGet()).append(ls).append(e.getConnectionStatistics().getInfo()).append(ls));
        log.debug(sb.toString());
    }

    private void printConnectedPeers() {
        if (!this.networkNode.getConfirmedConnections().isEmpty()) {
            StringBuilder result = new StringBuilder("\n\n------------------------------------------------------------\nConnected peers for node " + String.valueOf(this.networkNode.getNodeAddress()) + ":");
            this.networkNode.getConfirmedConnections().forEach(e -> result.append("\n").append(e.getPeersNodeAddressOptional()).append(" ").append((Object)e.getConnectionState().getPeerType()));
            result.append("\n------------------------------------------------------------\n");
            log.debug(result.toString());
        }
    }

    public Set<Peer> getReportedPeers() {
        return this.reportedPeers;
    }

    public int getMinConnections() {
        return this.minConnections;
    }

    public int getPeakNumConnections() {
        return this.peakNumConnections;
    }

    public int getNumAllConnectionsLostEvents() {
        return this.numAllConnectionsLostEvents;
    }

    public static interface Listener {
        public void onAllConnectionsLost();

        public void onNewConnectionAfterAllConnectionsLost();

        public void onAwakeFromStandby();
    }
}

