/*
 * Decompiled with CFR 0.152.
 */
package monero.common;

import com.fasterxml.jackson.core.type.TypeReference;
import common.utils.GenUtils;
import common.utils.JsonUtils;
import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.net.URI;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import monero.common.MoneroError;
import monero.common.MoneroRpcConnection;
import monero.daemon.model.MoneroNetworkType;
import monero.daemon.model.MoneroTx;
import monero.wallet.model.MoneroAddressType;
import monero.wallet.model.MoneroDecodedAddress;
import monero.wallet.model.MoneroIntegratedAddress;
import monero.wallet.model.MoneroTxConfig;
import org.apache.commons.codec.binary.Hex;
import org.bouncycastle.jcajce.provider.digest.Keccak;

public class MoneroUtils {
    private static final BigInteger XMR_AU_MULTIPLIER = new BigInteger("1000000000000");
    public static final int RING_SIZE = 12;
    private static int LOG_LEVEL = 0;
    private static long AU_PER_XMR = 1000000000000L;
    private static final int NUM_MNEMONIC_WORDS = 25;
    private static final int VIEW_KEY_LENGTH = 64;
    private static final String ALPHABET = "123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz";
    private static final List<Character> CHARS = new ArrayList<Character>();
    private static final BigDecimal ALPHABET_SIZE;
    private static final Map<Integer, Integer> ENCODED_BLOCK_SIZE;
    private static final int FULL_BLOCK_SIZE = 8;
    private static final int FULL_ENCODED_BLOCK_SIZE = 11;
    private static final BigDecimal UINT64_MAX;
    private static final Pattern STANDARD_ADDRESS_PATTERN;
    private static final Pattern INTEGRATED_ADDRESS_PATTERN;

    public static String getVersion() {
        return "0.8.36";
    }

    public static void tryLoadNativeLibrary() {
        if (!MoneroUtils.isNativeLibraryLoaded()) {
            try {
                MoneroUtils.loadNativeLibrary();
            }
            catch (Exception | UnsatisfiedLinkError throwable) {
                // empty catch block
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static void loadNativeLibrary() {
        String[] osName23322;
        if (MoneroUtils.isNativeLibraryLoaded()) {
            return;
        }
        try {
            String libName = (System.getProperty("os.name").toLowerCase().contains("windows") ? "lib" : "") + "monero-java";
            System.loadLibrary(libName);
            if (MoneroUtils.isNativeLibraryLoaded()) {
                return;
            }
        }
        catch (Exception | UnsatisfiedLinkError libName) {
            // empty catch block
        }
        Path tempDir = null;
        String[] libraryFiles = null;
        try {
            osName23322 = System.getProperty("os.name").toLowerCase();
            String osArch = System.getProperty("os.arch").toLowerCase();
            String libraryPath = "/";
            String libraryCppFile = null;
            String libraryJavaFile = null;
            if (osName23322.contains("windows")) {
                libraryPath = libraryPath + "windows/";
                libraryFiles = new String[]{"libmonero-cpp.dll", "libmonero-cpp.dll.a", "libmonero-java.dll", "libmonero-java.dll.a"};
                libraryCppFile = "libmonero-cpp.dll";
                libraryJavaFile = "libmonero-java.dll";
            } else if (osName23322.contains("linux")) {
                libraryPath = libraryPath + (osArch.contains("aarch64") ? "linux-arm64/" : "linux-x86_64/");
                libraryFiles = new String[]{"libmonero-cpp.so", "libmonero-java.so"};
                libraryCppFile = "libmonero-cpp.so";
                libraryJavaFile = "libmonero-java.so";
            } else {
                if (!osName23322.contains("mac")) throw new MoneroError("Unsupported operating system: " + (String)osName23322);
                libraryPath = libraryPath + (osArch.contains("aarch64") ? "mac-arm64/" : "mac-x86_64/");
                libraryFiles = new String[]{"libmonero-cpp.dylib", "libmonero-java.dylib"};
                libraryCppFile = "libmonero-cpp.dylib";
                libraryJavaFile = "libmonero-java.dylib";
            }
            tempDir = Files.createTempDirectory("libmonero", new FileAttribute[0]);
            for (String libraryFile : libraryFiles) {
                try (InputStream inputStream = MoneroUtils.class.getResourceAsStream(libraryPath + libraryFile);
                     OutputStream outputStream = Files.newOutputStream(tempDir.resolve(libraryFile), new OpenOption[0]);){
                    int bytesRead;
                    if (inputStream == null) {
                        throw new MoneroError("Missing native library for monero-java: " + libraryFile);
                    }
                    byte[] buffer = new byte[1024];
                    while ((bytesRead = inputStream.read(buffer)) != -1) {
                        outputStream.write(buffer, 0, bytesRead);
                    }
                }
            }
            System.load(tempDir.resolve(libraryCppFile).toString());
            System.load(tempDir.resolve(libraryJavaFile).toString());
        }
        catch (Exception | UnsatisfiedLinkError e) {
            try {
                throw new MoneroError(e);
            }
            catch (Throwable throwable) {
                try {
                    if (tempDir == null) throw throwable;
                    String[] stringArray = libraryFiles;
                    int n = stringArray.length;
                    int n2 = 0;
                    while (true) {
                        if (n2 >= n) {
                            Files.delete(tempDir);
                            throw throwable;
                        }
                        void var19_29 = stringArray[n2];
                        Files.delete(tempDir.resolve((String)var19_29));
                        ++n2;
                    }
                }
                catch (Exception exception) {
                    // empty catch block
                }
                throw throwable;
            }
        }
        try {
            if (tempDir == null) return;
            osName23322 = libraryFiles;
            int n = osName23322.length;
            int n3 = 0;
            while (true) {
                if (n3 >= n) {
                    Files.delete(tempDir);
                    return;
                }
                String libraryFile = osName23322[n3];
                Files.delete(tempDir.resolve(libraryFile));
                ++n3;
            }
        }
        catch (Exception osName23322) {
            return;
        }
    }

    public static boolean isNativeLibraryLoaded() {
        try {
            MoneroUtils.jsonToBinaryJni(JsonUtils.serialize(new HashMap()));
            return true;
        }
        catch (Exception | UnsatisfiedLinkError e) {
            return false;
        }
    }

    public static boolean walletExists(String path) {
        String basePath = path.lastIndexOf(46) > path.lastIndexOf(File.separatorChar) ? path.substring(0, path.lastIndexOf(46)) : path;
        File keysFile = new File(basePath + ".keys");
        return keysFile.exists() && keysFile.isFile();
    }

    public static void validateMnemonic(String mnemonic) {
        GenUtils.assertNotNull("Mnemonic phrase is not initialized", mnemonic);
        GenUtils.assertFalse("Mnemonic phrase is empty", mnemonic.isEmpty());
        String[] words = mnemonic.split(" ");
        if (words.length != 25) {
            throw new Error("Mnemonic phrase is " + words.length + " words but must be " + 25);
        }
    }

    public static boolean isValidPrivateViewKey(String privateViewKey) {
        try {
            MoneroUtils.validatePrivateViewKey(privateViewKey);
            return true;
        }
        catch (Exception e) {
            return false;
        }
    }

    public static boolean isValidPublicViewKey(String publicViewKey) {
        try {
            MoneroUtils.validatePublicViewKey(publicViewKey);
            return true;
        }
        catch (Exception e) {
            return false;
        }
    }

    public static boolean isValidPrivateSpendKey(String privateSpendKey) {
        try {
            MoneroUtils.validatePrivateSpendKey(privateSpendKey);
            return true;
        }
        catch (Exception e) {
            return false;
        }
    }

    public static boolean isValidPublicSpendKey(String publicSpendKey) {
        try {
            MoneroUtils.validatePublicSpendKey(publicSpendKey);
            return true;
        }
        catch (Exception e) {
            return false;
        }
    }

    public static void validatePrivateViewKey(String privateViewKey) {
        if (!MoneroUtils.isHex64(privateViewKey)) {
            throw new MoneroError("private view key expected to be 64 hex characters");
        }
    }

    public static void validatePublicViewKey(String publicViewKey) {
        if (!MoneroUtils.isHex64(publicViewKey)) {
            throw new MoneroError("public view key expected to be 64 hex characters");
        }
    }

    public static void validatePrivateSpendKey(String privateSpendKey) {
        if (!MoneroUtils.isHex64(privateSpendKey)) {
            throw new MoneroError("private spend key expected to be 64 hex characters");
        }
    }

    public static void validatePublicSpendKey(String publicSpendKey) {
        if (!MoneroUtils.isHex64(publicSpendKey)) {
            throw new MoneroError("public spend key expected to be 64 hex characters");
        }
    }

    public static MoneroIntegratedAddress getIntegratedAddress(MoneroNetworkType networkType, String standardAddress, String paymentId) {
        MoneroUtils.loadNativeLibrary();
        try {
            return JsonUtils.deserialize(MoneroUtils.getIntegratedAddressJni(networkType.ordinal(), standardAddress, paymentId == null ? "" : paymentId), MoneroIntegratedAddress.class);
        }
        catch (Exception err) {
            throw new MoneroError(err.getMessage());
        }
    }

    public static MoneroDecodedAddress decodeAddress(String address) {
        GenUtils.assertNotNull("Address is null", address);
        boolean isIntegrated = false;
        if (!STANDARD_ADDRESS_PATTERN.matcher(address).matches()) {
            if (INTEGRATED_ADDRESS_PATTERN.matcher(address).matches()) {
                isIntegrated = true;
            } else {
                throw new MoneroError("Address has invalid regex pattern");
            }
        }
        String addressHex = MoneroUtils.decodeAddressToHex(address);
        GenUtils.assertTrue("Address has invalid hash", MoneroUtils.isValidAddressHash(addressHex));
        int addressCode = Integer.parseInt(addressHex.substring(0, 2), 16);
        MoneroAddressType addressType = null;
        MoneroNetworkType networkType = null;
        for (MoneroNetworkType aNetworkType : MoneroNetworkType.values()) {
            if (addressCode == aNetworkType.getPrimaryAddressCode()) {
                GenUtils.assertFalse("Address has primary address code but integrated address pattern", isIntegrated);
                addressType = MoneroAddressType.PRIMARY_ADDRESS;
                networkType = aNetworkType;
                break;
            }
            if (addressCode == aNetworkType.getIntegratedAddressCode()) {
                GenUtils.assertTrue("Address has integrated address code but non-integrated address pattern", isIntegrated);
                addressType = MoneroAddressType.INTEGRATED_ADDRESS;
                networkType = aNetworkType;
                break;
            }
            if (addressCode != aNetworkType.getSubaddressCode()) continue;
            GenUtils.assertFalse("Address has subaddress code but integrated address pattern", isIntegrated);
            addressType = MoneroAddressType.SUBADDRESS;
            networkType = aNetworkType;
            break;
        }
        GenUtils.assertTrue("Address has invalid code: " + addressCode, addressType != null && networkType != null);
        return new MoneroDecodedAddress(address, addressType, networkType);
    }

    public static boolean isValidAddress(String address, MoneroNetworkType networkType) {
        try {
            MoneroUtils.validateAddress(address, networkType);
            return true;
        }
        catch (MoneroError e) {
            return false;
        }
    }

    public static void validateAddress(String address, MoneroNetworkType networkType) {
        try {
            MoneroDecodedAddress decodedAddress = MoneroUtils.decodeAddress(address);
            GenUtils.assertEquals("Address network type mismatch: " + (Object)((Object)networkType) + " vs " + (Object)((Object)decodedAddress.getNetworkType()), (Object)networkType, (Object)decodedAddress.getNetworkType());
        }
        catch (AssertionError e) {
            throw new MoneroError(((Throwable)((Object)e)).getMessage());
        }
    }

    public static void validatePaymentId(String paymentId) {
        GenUtils.assertTrue(paymentId.length() == 16 || paymentId.length() == 64);
    }

    public static void validateViewKey(String viewKey) {
        if (viewKey == null) {
            throw new MoneroError("View key is null");
        }
        if (viewKey.length() != 64) {
            throw new MoneroError("View key is " + viewKey.length() + " characters but must be " + 64);
        }
    }

    public static URI parseUri(String uri) {
        if (uri != null && uri.length() > 0 && !uri.toLowerCase().matches("^\\w+://.+")) {
            uri = "http://" + uri;
        }
        try {
            return new URI(uri);
        }
        catch (Exception e) {
            throw new MoneroError(e);
        }
    }

    public static void validateHex(String str) {
        if (!str.matches("^([0-9A-Fa-f]{2})+$")) {
            throw new MoneroError("Invalid hex: " + str);
        }
    }

    public static void validateBase58(String standardAddress) {
        for (char c : standardAddress.toCharArray()) {
            if (CHARS.contains(Character.valueOf(c))) continue;
            throw new MoneroError("Invalid Base58 " + standardAddress);
        }
    }

    public static boolean paymentIdsEqual(String paymentId1, String paymentId2) {
        int maxLength = Math.max(paymentId1.length(), paymentId2.length());
        for (int i = 0; i < maxLength; ++i) {
            if (i < paymentId1.length() && i < paymentId2.length() && paymentId1.charAt(i) != paymentId2.charAt(i)) {
                return false;
            }
            if (i >= paymentId1.length() && paymentId2.charAt(i) != '0') {
                return false;
            }
            if (i < paymentId2.length() || paymentId1.charAt(i) == '0') continue;
            return false;
        }
        return true;
    }

    public static <T extends MoneroTx> void mergeTx(List<T> txs, T tx) {
        for (MoneroTx aTx : txs) {
            if (!aTx.getHash().equals(tx.getHash())) continue;
            aTx.merge(tx);
            return;
        }
        txs.add(tx);
    }

    public static byte[] mapToBinary(Map<String, Object> map) {
        MoneroUtils.loadNativeLibrary();
        return MoneroUtils.jsonToBinaryJni(JsonUtils.serialize(map));
    }

    public static Map<String, Object> binaryToMap(byte[] bin) {
        MoneroUtils.loadNativeLibrary();
        return JsonUtils.deserialize(MoneroUtils.binaryToJsonJni(bin), new TypeReference<Map<String, Object>>(){});
    }

    public static Map<String, Object> binaryBlocksToMap(byte[] binBlocks) {
        MoneroUtils.loadNativeLibrary();
        Map<String, Object> map = JsonUtils.deserialize(MoneroRpcConnection.MAPPER, MoneroUtils.binaryBlocksToJsonJni(binBlocks), new TypeReference<Map<String, Object>>(){});
        ArrayList<Map<String, Object>> blockMaps = new ArrayList<Map<String, Object>>();
        for (String blockStr : (List)map.get("blocks")) {
            blockMaps.add(JsonUtils.deserialize(MoneroRpcConnection.MAPPER, blockStr, new TypeReference<Map<String, Object>>(){}));
        }
        map.put("blocks", blockMaps);
        ArrayList allTxs = new ArrayList();
        List rpcAllTxs = (List)map.get("txs");
        for (Object rpcTxs : rpcAllTxs) {
            if ("".equals(rpcTxs)) {
                allTxs.add(new ArrayList());
                continue;
            }
            ArrayList<Map<String, Object>> txs = new ArrayList<Map<String, Object>>();
            allTxs.add(txs);
            for (String rpcTx : (List)rpcTxs) {
                txs.add(JsonUtils.deserialize(MoneroRpcConnection.MAPPER, rpcTx.replaceFirst(",", "{") + "}", new TypeReference<Map<String, Object>>(){}));
            }
        }
        map.put("txs", allTxs);
        return map;
    }

    public static void log(int level, String msg) {
        GenUtils.assertTrue("Log level must be an integer >= 0", level >= 0);
        if (LOG_LEVEL >= level) {
            System.out.println(msg);
        }
    }

    public static void setLogLevel(int level) {
        GenUtils.assertTrue("Log level must be an integer >= 0", level >= 0);
        LOG_LEVEL = level;
        if (MoneroUtils.isNativeLibraryLoaded()) {
            MoneroUtils.setLogLevelJni(level);
        }
    }

    public static int getLogLevel() {
        return LOG_LEVEL;
    }

    public static void configureNativeLogging(String path, boolean console) {
        MoneroUtils.loadNativeLibrary();
        MoneroUtils.configureLoggingJni(path, console);
    }

    public static BigInteger xmrToAtomicUnits(double amountXmr) {
        return new BigDecimal(amountXmr).multiply(new BigDecimal(XMR_AU_MULTIPLIER)).setScale(0, RoundingMode.HALF_UP).toBigInteger();
    }

    public static double atomicUnitsToXmr(BigInteger amountAtomicUnits) {
        return new BigDecimal(amountAtomicUnits).divide(new BigDecimal(XMR_AU_MULTIPLIER), 12, RoundingMode.HALF_UP).doubleValue();
    }

    public static double divide(BigInteger auDividend, BigInteger auDivisor) {
        return new BigDecimal(auDividend).divide(new BigDecimal(auDivisor), 12, RoundingMode.HALF_UP).doubleValue();
    }

    public static BigInteger multiply(BigInteger amount1, double amount2) {
        return amount1 == null ? null : new BigDecimal(amount1).multiply(BigDecimal.valueOf(amount2)).setScale(0, RoundingMode.HALF_UP).toBigInteger();
    }

    public static String getPaymentUri(MoneroTxConfig config) {
        if (config.getAddress() == null) {
            throw new IllegalArgumentException("Payment URI requires an address");
        }
        StringBuilder sb = new StringBuilder();
        sb.append("monero:");
        sb.append(config.getAddress());
        StringBuilder paramSb = new StringBuilder();
        if (config.getAmount() != null) {
            paramSb.append("&tx_amount=").append(MoneroUtils.atomicUnitsToXmr(config.getAmount()));
        }
        if (config.getRecipientName() != null) {
            paramSb.append("&recipient_name=").append(MoneroUtils.urlEncode(config.getRecipientName()));
        }
        if (config.getNote() != null) {
            paramSb.append("&tx_description=").append(MoneroUtils.urlEncode(config.getNote()));
        }
        if (config.getPaymentId() != null && config.getPaymentId().length() > 0) {
            throw new IllegalArgumentException("Standalone payment id deprecated, use integrated address instead");
        }
        char[] paramChars = paramSb.toString().toCharArray();
        if (paramChars.length > 0) {
            paramChars[0] = 63;
        }
        return sb.toString() + (paramChars.length > 0 ? new String(paramChars) : "");
    }

    private static String urlEncode(String string) {
        return URLEncoder.encode(string, StandardCharsets.UTF_8).replaceAll("\\+", "%20");
    }

    private static native String getIntegratedAddressJni(int var0, String var1, String var2);

    private static native byte[] jsonToBinaryJni(String var0);

    private static native String binaryToJsonJni(byte[] var0);

    private static native String binaryBlocksToJsonJni(byte[] var0);

    private static native void configureLoggingJni(String var0, boolean var1);

    private static native void setLogLevelJni(int var0);

    private static boolean isHex64(String str) {
        return str != null && str.length() == 64 && GenUtils.isHex(str);
    }

    private static boolean isValidAddressHash(String decodedAddrStr) {
        String checksumCheck = decodedAddrStr.substring(decodedAddrStr.length() - 8);
        Keccak.Digest256 digest256 = new Keccak.Digest256();
        String withoutChecksumStr = decodedAddrStr.substring(0, decodedAddrStr.length() - 8);
        byte[] withoutChecksumBytes = MoneroUtils.hexToBin(withoutChecksumStr);
        byte[] hashbytes = digest256.digest(withoutChecksumBytes);
        String encodedStr = new String(Hex.encodeHex(hashbytes));
        String hashChecksum = encodedStr.substring(0, 8);
        return hashChecksum != null && hashChecksum.equals(checksumCheck);
    }

    private static String decodeAddressToHex(String address) {
        int[] bin = new int[address.length()];
        for (int i = 0; i < address.length(); ++i) {
            bin[i] = address.codePointAt(i);
        }
        int fullBlockCount = (int)Math.floor(bin.length / 11);
        int lastBlockSize = bin.length % 11;
        int lastBlockDecodedSize = ENCODED_BLOCK_SIZE.get(lastBlockSize);
        if (lastBlockDecodedSize < 0) {
            throw new IllegalArgumentException("Invalid encoded length");
        }
        int dataSize = fullBlockCount * 8 + lastBlockDecodedSize;
        int[] data = new int[dataSize];
        for (int i = 0; i < fullBlockCount; ++i) {
            data = MoneroUtils.decodeBlock(GenUtils.subarray(bin, i * 11, i * 11 + 11), data, i * 8);
        }
        if (lastBlockSize > 0) {
            int[] subarray = GenUtils.subarray(bin, fullBlockCount * 11, fullBlockCount * 11 + 8);
            data = MoneroUtils.decodeBlock(subarray, data, fullBlockCount * 8);
        }
        return MoneroUtils.binToHex(data);
    }

    private static int[] decodeBlock(int[] data, int[] buf, int index) {
        if (data.length < 1 || data.length > 11) {
            throw new RuntimeException("Invalid block length: " + data.length);
        }
        int resSize = ENCODED_BLOCK_SIZE.get(data.length);
        if (resSize <= 0) {
            throw new RuntimeException("Invalid block size");
        }
        BigDecimal resNum = BigDecimal.ZERO;
        BigDecimal order = BigDecimal.ONE;
        for (int i = data.length - 1; i >= 0; --i) {
            int digit = ALPHABET.indexOf(data[i]);
            if (digit < 0) {
                throw new RuntimeException("Invalid symbol");
            }
            BigDecimal product = order.multiply(new BigDecimal(digit)).add(resNum);
            if (product.compareTo(UINT64_MAX) > 0) {
                throw new RuntimeException("Overflow");
            }
            resNum = product;
            order = order.multiply(ALPHABET_SIZE);
        }
        if (resSize < 8 && new BigDecimal(2).pow(8 * resSize).compareTo(resNum) <= 0) {
            throw new RuntimeException("Overflow 2");
        }
        int[] tmpBuf = MoneroUtils.uint64To8be(resNum, resSize);
        for (int j = 0; j < tmpBuf.length; ++j) {
            buf[j + index] = tmpBuf[j];
        }
        return buf;
    }

    private static int[] uint64To8be(BigDecimal num, int size) {
        int[] res = new int[size];
        if (size < 1 || size > 8) {
            throw new RuntimeException("Invalid input length");
        }
        BigDecimal twopow8 = new BigDecimal(2).pow(8);
        for (int i = size - 1; i >= 0; --i) {
            res[i] = num.remainder(twopow8).intValue();
            num = num.divide(twopow8);
        }
        return res;
    }

    private static byte[] hexToBin(String hexStr) {
        if (hexStr == null || hexStr.length() % 2 != 0) {
            return null;
        }
        byte[] res = new byte[hexStr.length() / 2];
        for (int i = 0; i < hexStr.length() / 2; ++i) {
            res[i] = (byte)Integer.parseInt(hexStr.substring(i * 2, i * 2 + 2), 16);
        }
        return res;
    }

    private static String binToHex(int[] data) {
        StringBuilder builder = new StringBuilder();
        for (int i : data) {
            builder.append(String.format("%02x", i));
        }
        return builder.toString();
    }

    static {
        for (char c : ALPHABET.toCharArray()) {
            CHARS.add(Character.valueOf(c));
        }
        ALPHABET_SIZE = new BigDecimal(ALPHABET.length());
        ENCODED_BLOCK_SIZE = new HashMap<Integer, Integer>();
        ENCODED_BLOCK_SIZE.put(0, 0);
        ENCODED_BLOCK_SIZE.put(2, 1);
        ENCODED_BLOCK_SIZE.put(3, 2);
        ENCODED_BLOCK_SIZE.put(5, 3);
        ENCODED_BLOCK_SIZE.put(6, 4);
        ENCODED_BLOCK_SIZE.put(7, 5);
        ENCODED_BLOCK_SIZE.put(9, 6);
        ENCODED_BLOCK_SIZE.put(10, 7);
        ENCODED_BLOCK_SIZE.put(11, 8);
        UINT64_MAX = new BigDecimal(Math.pow(2.0, 64.0));
        STANDARD_ADDRESS_PATTERN = Pattern.compile("^[123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz]{95}$");
        INTEGRATED_ADDRESS_PATTERN = Pattern.compile("^[123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz]{106}$");
    }
}

