/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.core;

import com.google.common.base.Preconditions;
import com.google.common.hash.HashCode;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import com.google.common.io.BaseEncoding;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.util.Arrays;
import java.util.Map;
import java.util.TreeMap;
import javax.annotation.Nullable;
import org.bitcoinj.core.Block;
import org.bitcoinj.core.Context;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.ProtocolException;
import org.bitcoinj.core.Sha256Hash;
import org.bitcoinj.core.StoredBlock;
import org.bitcoinj.core.Utils;
import org.bitcoinj.core.VerificationException;
import org.bitcoinj.store.BlockStore;
import org.bitcoinj.store.BlockStoreException;
import org.bitcoinj.store.FullPrunedBlockStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CheckpointManager {
    private static final Logger log = LoggerFactory.getLogger(CheckpointManager.class);
    private static final String BINARY_MAGIC = "CHECKPOINTS 1";
    private static final String TEXTUAL_MAGIC = "TXT CHECKPOINTS 1";
    private static final int MAX_SIGNATURES = 256;
    protected final TreeMap<Long, StoredBlock> checkpoints = new TreeMap();
    protected final NetworkParameters params;
    protected final Sha256Hash dataHash;
    public static final BaseEncoding BASE64 = BaseEncoding.base64().omitPadding();

    public CheckpointManager(Context context) throws IOException {
        this(context.getParams(), null);
    }

    public CheckpointManager(NetworkParameters params, @Nullable InputStream inputStream) throws IOException {
        this.params = Preconditions.checkNotNull(params);
        if (inputStream == null) {
            inputStream = CheckpointManager.openStream(params);
        }
        Preconditions.checkNotNull(inputStream);
        inputStream = new BufferedInputStream(inputStream);
        inputStream.mark(1);
        int first = inputStream.read();
        inputStream.reset();
        if (first == BINARY_MAGIC.charAt(0)) {
            this.dataHash = this.readBinary(inputStream);
        } else if (first == TEXTUAL_MAGIC.charAt(0)) {
            this.dataHash = this.readTextual(inputStream);
        } else {
            throw new IOException("Unsupported format.");
        }
    }

    public static InputStream openStream(NetworkParameters params) {
        return CheckpointManager.class.getResourceAsStream("/" + params.getId() + ".checkpoints.txt");
    }

    private Sha256Hash readBinary(InputStream inputStream) throws IOException {
        FilterInputStream dis = null;
        try {
            MessageDigest digest = Sha256Hash.newDigest();
            DigestInputStream digestInputStream = new DigestInputStream(inputStream, digest);
            dis = new DataInputStream(digestInputStream);
            digestInputStream.on(false);
            byte[] header = new byte[BINARY_MAGIC.length()];
            ((DataInputStream)dis).readFully(header);
            if (!Arrays.equals(header, BINARY_MAGIC.getBytes(StandardCharsets.US_ASCII))) {
                throw new IOException("Header bytes did not match expected version");
            }
            int numSignatures = Preconditions.checkPositionIndex(((DataInputStream)dis).readInt(), 256, "Num signatures out of range");
            for (int i = 0; i < numSignatures; ++i) {
                byte[] sig = new byte[65];
                ((DataInputStream)dis).readFully(sig);
            }
            digestInputStream.on(true);
            int numCheckpoints = ((DataInputStream)dis).readInt();
            Preconditions.checkState(numCheckpoints > 0);
            int size = 96;
            ByteBuffer buffer = ByteBuffer.allocate(96);
            for (int i = 0; i < numCheckpoints; ++i) {
                if (((DataInputStream)dis).read(buffer.array(), 0, 96) < 96) {
                    throw new IOException("Incomplete read whilst loading checkpoints.");
                }
                StoredBlock block = StoredBlock.deserializeCompact(this.params, buffer);
                buffer.position(0);
                this.checkpoints.put(block.getHeader().getTimeSeconds(), block);
            }
            Sha256Hash dataHash = Sha256Hash.wrap(digest.digest());
            log.info("Read {} checkpoints up to time {}, hash is {}", this.checkpoints.size(), Utils.dateTimeFormat(this.checkpoints.lastEntry().getKey() * 1000L), dataHash);
            Sha256Hash sha256Hash = dataHash;
            return sha256Hash;
        }
        catch (ProtocolException e) {
            throw new IOException(e);
        }
        finally {
            if (dis != null) {
                dis.close();
            }
            inputStream.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Sha256Hash readTextual(InputStream inputStream) throws IOException {
        Hasher hasher = Hashing.sha256().newHasher();
        try (BufferedReader reader = null;){
            reader = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.US_ASCII));
            String magic = reader.readLine();
            if (!TEXTUAL_MAGIC.equals(magic)) {
                throw new IOException("unexpected magic: " + magic);
            }
            int numSigs = Integer.parseInt(reader.readLine());
            for (int i = 0; i < numSigs; ++i) {
                reader.readLine();
            }
            int numCheckpoints = Integer.parseInt(reader.readLine());
            Preconditions.checkState(numCheckpoints > 0);
            hasher.putBytes(ByteBuffer.allocate(4).order(ByteOrder.BIG_ENDIAN).putInt(numCheckpoints).array());
            int size = 96;
            ByteBuffer buffer = ByteBuffer.allocate(96);
            for (int i = 0; i < numCheckpoints; ++i) {
                byte[] bytes = BASE64.decode(reader.readLine());
                hasher.putBytes(bytes);
                buffer.position(0);
                buffer.put(bytes);
                buffer.position(0);
                StoredBlock block = StoredBlock.deserializeCompact(this.params, buffer);
                this.checkpoints.put(block.getHeader().getTimeSeconds(), block);
            }
            HashCode hash = hasher.hash();
            log.info("Read {} checkpoints up to time {}, hash is {}", this.checkpoints.size(), Utils.dateTimeFormat(this.checkpoints.lastEntry().getKey() * 1000L), hash);
            Sha256Hash sha256Hash = Sha256Hash.wrap(hash.asBytes());
            return sha256Hash;
        }
    }

    public StoredBlock getCheckpointBefore(long timeSecs) {
        try {
            Preconditions.checkArgument(timeSecs > this.params.getGenesisBlock().getTimeSeconds());
            Map.Entry<Long, StoredBlock> entry = this.checkpoints.floorEntry(timeSecs);
            if (entry != null) {
                return entry.getValue();
            }
            Block genesis = this.params.getGenesisBlock().cloneAsHeader();
            return new StoredBlock(genesis, genesis.getWork(), 0);
        }
        catch (VerificationException e) {
            throw new RuntimeException(e);
        }
    }

    public int numCheckpoints() {
        return this.checkpoints.size();
    }

    public Sha256Hash getDataHash() {
        return this.dataHash;
    }

    public static void checkpoint(NetworkParameters params, InputStream checkpoints, BlockStore store, long timeSecs) throws IOException, BlockStoreException {
        Preconditions.checkNotNull(params);
        Preconditions.checkNotNull(store);
        Preconditions.checkArgument(!(store instanceof FullPrunedBlockStore), "You cannot use checkpointing with a full store.");
        Preconditions.checkArgument((timeSecs -= 604800L) > 0L);
        log.info("Attempting to initialize a new block store with a checkpoint for time {} ({})", (Object)timeSecs, (Object)Utils.dateTimeFormat(timeSecs * 1000L));
        BufferedInputStream stream = new BufferedInputStream(checkpoints);
        CheckpointManager manager = new CheckpointManager(params, stream);
        StoredBlock checkpoint = manager.getCheckpointBefore(timeSecs);
        store.put(checkpoint);
        store.setChainHead(checkpoint);
    }
}

