/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.crypto;

import com.google.common.base.Preconditions;
import com.google.common.primitives.Bytes;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.text.Normalizer;
import java.util.Arrays;
import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import org.bitcoinj.core.AddressFormatException;
import org.bitcoinj.core.Base58;
import org.bitcoinj.core.ECKey;
import org.bitcoinj.core.LegacyAddress;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.PrefixedChecksummedBytes;
import org.bitcoinj.core.Sha256Hash;
import org.bitcoinj.crypto.DRMWorkaround;
import org.bouncycastle.crypto.generators.SCrypt;

public class BIP38PrivateKey
extends PrefixedChecksummedBytes {
    public final boolean ecMultiply;
    public final boolean compressed;
    public final boolean hasLotAndSequence;
    public final byte[] addressHash;
    public final byte[] content;

    public static BIP38PrivateKey fromBase58(NetworkParameters params, String base58) throws AddressFormatException {
        boolean ecMultiply;
        boolean compressed;
        byte[] versionAndDataBytes = Base58.decodeChecked(base58);
        int version = versionAndDataBytes[0] & 0xFF;
        byte[] bytes = Arrays.copyOfRange(versionAndDataBytes, 1, versionAndDataBytes.length);
        if (version != 1) {
            throw new AddressFormatException.InvalidPrefix("Mismatched version number: " + version);
        }
        if (bytes.length != 38) {
            throw new AddressFormatException.InvalidDataLength("Wrong number of bytes: " + bytes.length);
        }
        boolean hasLotAndSequence = (bytes[1] & 4) != 0;
        boolean bl = compressed = (bytes[1] & 0x20) != 0;
        if ((bytes[1] & 1) != 0) {
            throw new AddressFormatException("Bit 0x01 reserved for future use.");
        }
        if ((bytes[1] & 2) != 0) {
            throw new AddressFormatException("Bit 0x02 reserved for future use.");
        }
        if ((bytes[1] & 8) != 0) {
            throw new AddressFormatException("Bit 0x08 reserved for future use.");
        }
        if ((bytes[1] & 0x10) != 0) {
            throw new AddressFormatException("Bit 0x10 reserved for future use.");
        }
        int byte0 = bytes[0] & 0xFF;
        if (byte0 == 66) {
            if ((bytes[1] & 0xC0) != 192) {
                throw new AddressFormatException("Bits 0x40 and 0x80 must be set for non-EC-multiplied keys.");
            }
            ecMultiply = false;
            if (hasLotAndSequence) {
                throw new AddressFormatException("Non-EC-multiplied keys cannot have lot/sequence.");
            }
        } else if (byte0 == 67) {
            if ((bytes[1] & 0xC0) != 0) {
                throw new AddressFormatException("Bits 0x40 and 0x80 must be cleared for EC-multiplied keys.");
            }
            ecMultiply = true;
        } else {
            throw new AddressFormatException("Second byte must by 0x42 or 0x43.");
        }
        byte[] addressHash = Arrays.copyOfRange(bytes, 2, 6);
        byte[] content = Arrays.copyOfRange(bytes, 6, 38);
        return new BIP38PrivateKey(params, bytes, ecMultiply, compressed, hasLotAndSequence, addressHash, content);
    }

    private BIP38PrivateKey(NetworkParameters params, byte[] bytes, boolean ecMultiply, boolean compressed, boolean hasLotAndSequence, byte[] addressHash, byte[] content) throws AddressFormatException {
        super(params, bytes);
        this.ecMultiply = ecMultiply;
        this.compressed = compressed;
        this.hasLotAndSequence = hasLotAndSequence;
        this.addressHash = addressHash;
        this.content = content;
    }

    public String toBase58() {
        return Base58.encodeChecked(1, this.bytes);
    }

    public ECKey decrypt(String passphrase) throws BadPassphraseException {
        String normalizedPassphrase = Normalizer.normalize(passphrase, Normalizer.Form.NFC);
        ECKey key = this.ecMultiply ? this.decryptEC(normalizedPassphrase) : this.decryptNoEC(normalizedPassphrase);
        Sha256Hash hash = Sha256Hash.twiceOf(LegacyAddress.fromKey(this.params, key).toString().getBytes(StandardCharsets.US_ASCII));
        byte[] actualAddressHash = Arrays.copyOfRange(hash.getBytes(), 0, 4);
        if (!Arrays.equals(actualAddressHash, this.addressHash)) {
            throw new BadPassphraseException();
        }
        return key;
    }

    private ECKey decryptNoEC(String normalizedPassphrase) {
        try {
            byte[] derived = SCrypt.generate(normalizedPassphrase.getBytes(StandardCharsets.UTF_8), this.addressHash, 16384, 8, 8, 64);
            byte[] key = Arrays.copyOfRange(derived, 32, 64);
            SecretKeySpec keyspec = new SecretKeySpec(key, "AES");
            DRMWorkaround.maybeDisableExportControls();
            Cipher cipher = Cipher.getInstance("AES/ECB/NoPadding");
            cipher.init(2, keyspec);
            byte[] decrypted = cipher.doFinal(this.content, 0, 32);
            for (int i = 0; i < 32; ++i) {
                int n = i;
                decrypted[n] = (byte)(decrypted[n] ^ derived[i]);
            }
            return ECKey.fromPrivate(decrypted, this.compressed);
        }
        catch (GeneralSecurityException x) {
            throw new RuntimeException(x);
        }
    }

    private ECKey decryptEC(String normalizedPassphrase) {
        try {
            byte[] ownerEntropy = Arrays.copyOfRange(this.content, 0, 8);
            byte[] ownerSalt = this.hasLotAndSequence ? Arrays.copyOfRange(ownerEntropy, 0, 4) : ownerEntropy;
            byte[] passFactorBytes = SCrypt.generate(normalizedPassphrase.getBytes(StandardCharsets.UTF_8), ownerSalt, 16384, 8, 8, 32);
            if (this.hasLotAndSequence) {
                byte[] hashBytes = Bytes.concat(passFactorBytes, ownerEntropy);
                Preconditions.checkState(hashBytes.length == 40);
                passFactorBytes = Sha256Hash.hashTwice(hashBytes);
            }
            BigInteger passFactor = new BigInteger(1, passFactorBytes);
            ECKey k = ECKey.fromPrivate(passFactor, true);
            byte[] salt = Bytes.concat(this.addressHash, ownerEntropy);
            Preconditions.checkState(salt.length == 12);
            byte[] derived = SCrypt.generate(k.getPubKey(), salt, 1024, 1, 1, 64);
            byte[] aeskey = Arrays.copyOfRange(derived, 32, 64);
            SecretKeySpec keyspec = new SecretKeySpec(aeskey, "AES");
            Cipher cipher = Cipher.getInstance("AES/ECB/NoPadding");
            cipher.init(2, keyspec);
            byte[] encrypted2 = Arrays.copyOfRange(this.content, 16, 32);
            byte[] decrypted2 = cipher.doFinal(encrypted2);
            Preconditions.checkState(decrypted2.length == 16);
            for (int i = 0; i < 16; ++i) {
                int n = i;
                decrypted2[n] = (byte)(decrypted2[n] ^ derived[i + 16]);
            }
            byte[] encrypted1 = Bytes.concat(Arrays.copyOfRange(this.content, 8, 16), Arrays.copyOfRange(decrypted2, 0, 8));
            byte[] decrypted1 = cipher.doFinal(encrypted1);
            Preconditions.checkState(decrypted1.length == 16);
            for (int i = 0; i < 16; ++i) {
                int n = i;
                decrypted1[n] = (byte)(decrypted1[n] ^ derived[i]);
            }
            byte[] seed = Bytes.concat(decrypted1, Arrays.copyOfRange(decrypted2, 8, 16));
            Preconditions.checkState(seed.length == 24);
            BigInteger seedFactor = new BigInteger(1, Sha256Hash.hashTwice(seed));
            Preconditions.checkState(passFactor.signum() >= 0);
            Preconditions.checkState(seedFactor.signum() >= 0);
            BigInteger priv = passFactor.multiply(seedFactor).mod(ECKey.CURVE.getN());
            return ECKey.fromPrivate(priv, this.compressed);
        }
        catch (GeneralSecurityException x) {
            throw new RuntimeException(x);
        }
    }

    public String toString() {
        return this.toBase58();
    }

    public static final class BadPassphraseException
    extends Exception {
    }
}

